<?php

namespace App\Http\Livewire\Production\Reports;

use Livewire\Component;
use App\Models\ShipmentLine;

class ExftyHistoryDropdown extends Component
{
    public $shipmentLineId = null;
    public $show = false;
    public $history = [];
    public $loading = false;
    public $position = ['left' => 0, 'top' => 0];

    protected $listeners = ['showExftyHistory' => 'showHistoryFromEvent', 'hideExftyHistory' => 'hideHistory'];

    public function showHistory($shipmentLineId, $left = 200, $top = 200)
    {
        \Log::info('ExftyHistoryDropdown: showHistory called with shipmentLineId:', ['shipmentLineId' => $shipmentLineId, 'left' => $left, 'top' => $top]);
        $this->shipmentLineId = $shipmentLineId;
        $this->position = ['left' => $left, 'top' => $top];
        $this->show = true;
        $this->loading = true;
        $this->history = [];
        
        // Load history data
        $this->loadHistory();
    }

    public function showHistoryFromEvent($shipmentLineId, $left = 200, $top = 200)
    {
        \Log::info('ExftyHistoryDropdown: showHistoryFromEvent called with shipmentLineId:', ['shipmentLineId' => $shipmentLineId, 'left' => $left, 'top' => $top]);
        $this->showHistory($shipmentLineId, $left, $top);
    }

    public function hideHistory()
    {
        $this->show = false;
        $this->shipmentLineId = null;
        $this->history = [];
        $this->loading = false;
    }

    private function loadHistory()
    {
        if (!$this->shipmentLineId) {
            return;
        }

        $this->loading = true;
        
        $history = \DB::table('audits')
            ->where('auditable_type', 'App\\Models\\ShipmentLine')
            ->where('auditable_id', $this->shipmentLineId)
            ->whereRaw("JSON_CONTAINS_PATH(old_values, 'one', '$.exfty')")
            ->select([
                'created_at',
                \DB::raw("JSON_UNQUOTE(JSON_EXTRACT(old_values, '$.exfty')) as old_exfty"),
                \DB::raw("JSON_UNQUOTE(JSON_EXTRACT(new_values, '$.exfty')) as new_exfty")
            ])
            ->orderBy('created_at', 'asc')
            ->get();

        // Add current exfty as the latest entry if it exists
        $currentExfty = ShipmentLine::find($this->shipmentLineId)?->exfty;
        if ($currentExfty) {
            $history->push((object) [
                'created_at' => now(),
                'old_exfty' => $history->last()?->new_exfty ?? null,
                'new_exfty' => $currentExfty
            ]);
        }

        $this->history = $history->toArray();
        $this->loading = false;
    }

    public function render()
    {
        return view('livewire.production.reports.exfty-history-dropdown');
    }
}
