@props(['track', 'autoplay' => false])

<div class="bg-white rounded-2xl shadow-lg p-6" x-data="audioPlayer(@js($track), @js($autoplay))">
    <!-- Track Info Header -->
    <div class="flex items-center justify-between mb-6">
        <div class="flex items-center space-x-4">
            <div class="w-16 h-16 bg-gradient-to-br from-primary-100 to-accent-100 rounded-xl flex items-center justify-center">
                <svg class="w-8 h-8 text-primary-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3"></path>
                </svg>
            </div>
            <div>
                <h3 class="text-lg font-semibold text-gray-900">{{ $track->title }}</h3>
                <p class="text-sm text-gray-600">{{ $track->category->name }} • {{ $track->difficulty_level }}</p>
            </div>
        </div>
        
        <!-- Audio Status Indicator -->
        <div class="flex items-center space-x-2">
            <div class="flex space-x-1">
                <template x-for="i in 3" :key="i">
                    <div class="w-1 h-4 bg-gray-200 rounded-full transition-all duration-300"
                         :class="{ 'bg-primary-500': isPlaying && Math.random() > 0.5 }"></div>
                </template>
            </div>
            <span class="text-sm text-gray-500" x-text="isPlaying ? 'Playing' : 'Paused'"></span>
        </div>
    </div>

    <!-- Audio Element (Hidden) -->
    <audio 
        x-ref="audio" 
        preload="metadata"
        @loadedmetadata="onLoadedMetadata"
        @timeupdate="onTimeUpdate"
        @ended="onEnded"
        @error="onError"
        @canplay="onCanPlay"
        @waiting="onWaiting"
        @playing="onPlaying"
        @pause="onPause"
        class="hidden"
    >
        @if($track->audio_url)
            <source src="{{ $track->audio_url }}" type="audio/mpeg">
        @endif
        Your browser does not support the audio element.
    </audio>

    <!-- Progress Bar -->
    <div class="mb-6">
        <div class="flex items-center justify-between text-sm text-gray-500 mb-2">
            <span x-text="formatTime(currentTime)"></span>
            <span x-text="formatTime(duration)"></span>
        </div>
        <div class="relative">
            <div class="w-full bg-gray-200 rounded-full h-2">
                <div class="bg-primary-500 h-2 rounded-full transition-all duration-100" 
                     :style="`width: ${progressPercentage}%`"></div>
            </div>
            <input 
                type="range" 
                x-ref="progressSlider"
                min="0" 
                max="100" 
                step="0.1"
                class="absolute inset-0 w-full h-2 opacity-0 cursor-pointer"
                @input="seekTo($event.target.value)"
            >
        </div>
    </div>

    <!-- Main Controls -->
    <div class="flex items-center justify-center space-x-6 mb-6">
        <!-- Play/Pause -->
        <button @click="togglePlay" class="p-8 bg-gradient-to-br from-primary-500 via-primary-600 to-primary-700 hover:from-primary-600 hover:via-primary-700 hover:to-primary-800 text-white rounded-full transition-all duration-300 transform hover:scale-110 hover:rotate-1 shadow-2xl hover:shadow-3xl border-4 border-white">
            <svg x-show="!isPlaying" class="w-16 h-16" fill="currentColor" viewBox="0 0 24 24">
                <path d="M8 5v14l11-7z"/>
            </svg>
            <svg x-show="isPlaying" class="w-16 h-16" fill="currentColor" viewBox="0 0 24 24">
                <path d="M6 19h4V5H6v14zm8-14v14h4V5h-4z"/>
            </svg>
        </button>

        <!-- Stop Button -->
        <button @click="stop" class="p-3 text-gray-700 hover:text-red-600 transition-colors bg-gray-100 hover:bg-red-100 rounded-full">
            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 6h12v12H6z"></path>
            </svg>
        </button>
    </div>

    <!-- Secondary Controls -->
    <div class="flex items-center justify-between">
        <!-- Volume Control -->
        <div class="flex items-center space-x-3">
            <svg class="w-5 h-5 text-gray-700" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15.536 8.464a5 5 0 010 7.072m2.828-9.9a9 9 0 010 12.728M5.586 15H4a1 1 0 01-1-1v-4a1 1 0 011-1h1.586l4.707-4.707C10.923 3.663 12 4.109 12 5v14c0 .891-1.077 1.337-1.707.707L5.586 15z"></path>
            </svg>
            <input 
                type="range" 
                min="0" 
                max="100" 
                x-model="volume" 
                @input="setVolume($event.target.value)"
                class="w-20 h-2 bg-gray-300 rounded-lg appearance-none cursor-pointer slider"
            >
            <span class="text-sm text-gray-700 w-8 font-medium" x-text="`${Math.round(volume)}%`"></span>
        </div>

        <!-- Playback Speed -->
        <div class="flex items-center space-x-2">
            <span class="text-sm text-gray-700 font-medium">Speed:</span>
            <select x-model="playbackRate" @change="setPlaybackRate" class="text-sm border-gray-300 rounded-md focus:ring-primary-500 focus:border-primary-500 bg-white text-gray-700">
                <option value="0.5">0.5x</option>
                <option value="0.75">0.75x</option>
                <option value="1" selected>1x</option>
                <option value="1.25">1.25x</option>
                <option value="1.5">1.5x</option>
                <option value="2">2x</option>
            </select>
        </div>

        <!-- Download Button -->
        @if($track->audio_url)
            <a href="{{ $track->audio_url }}" download="{{ $track->title }}.mp3" 
               class="inline-flex items-center px-3 py-2 text-sm font-medium text-gray-700 bg-gray-100 hover:bg-gray-200 rounded-lg transition-colors">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                Download
            </a>
        @endif
    </div>

    <!-- Loading State -->
    <div x-show="isLoading" class="mt-4 text-center">
        <div class="inline-flex items-center px-4 py-2 text-sm text-gray-600 bg-gray-100 rounded-lg">
            <svg class="animate-spin -ml-1 mr-3 h-4 w-4 text-primary-500" fill="none" viewBox="0 0 24 24">
                <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
            </svg>
            Loading audio...
        </div>
    </div>

    <!-- Error State -->
    <div x-show="error" class="mt-4 p-4 bg-red-50 border border-red-200 rounded-lg">
        <div class="flex items-center">
            <svg class="w-5 h-5 text-red-400 mr-2" fill="currentColor" viewBox="0 0 20 20">
                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"></path>
            </svg>
            <span class="text-sm text-red-700" x-text="error"></span>
        </div>
    </div>
</div>

<style>
.slider::-webkit-slider-thumb {
    appearance: none;
    height: 16px;
    width: 16px;
    border-radius: 50%;
    background: #4dbda0;
    cursor: pointer;
    border: 2px solid #ffffff;
    box-shadow: 0 2px 4px rgba(0,0,0,0.2);
}

.slider::-moz-range-thumb {
    height: 16px;
    width: 16px;
    border-radius: 50%;
    background: #4dbda0;
    cursor: pointer;
    border: 2px solid #ffffff;
    box-shadow: 0 2px 4px rgba(0,0,0,0.2);
}
</style>

<script>
function audioPlayer(track, autoplay) {
    return {
        track: track,
        audio: null,
        isPlaying: false,
        isLoading: true,
        error: null,
        currentTime: 0,
        duration: 0,
        volume: 100,
        playbackRate: 1,
        
        get progressPercentage() {
            if (this.duration === 0) return 0;
            return (this.currentTime / this.duration) * 100;
        },

        init() {
            this.audio = this.$refs.audio;
            if (this.audio) {
                this.audio.volume = this.volume / 100;
                this.audio.playbackRate = this.playbackRate;
                
                if (autoplay) {
                    this.play();
                }
            }
        },

        togglePlay() {
            if (this.isPlaying) {
                this.pause();
            } else {
                this.play();
            }
        },

        async play() {
            try {
                this.error = null;
                if (this.audio) {
                    await this.audio.play();
                }
            } catch (err) {
                this.error = 'Failed to play audio: ' + err.message;
                console.error('Audio play error:', err);
            }
        },

        pause() {
            if (this.audio) {
                this.audio.pause();
            }
        },

        stop() {
            if (this.audio) {
                this.audio.pause();
                this.audio.currentTime = 0;
                this.isPlaying = false;
                this.currentTime = 0;
            }
        },

        seek(seconds) {
            if (this.audio) {
                const newTime = this.currentTime + seconds;
                this.audio.currentTime = Math.max(0, Math.min(newTime, this.duration));
            }
        },

        seekTo(percentage) {
            if (this.audio) {
                const newTime = (percentage / 100) * this.duration;
                this.audio.currentTime = newTime;
            }
        },

        setVolume(value) {
            this.volume = value;
            if (this.audio) {
                this.audio.volume = value / 100;
            }
        },

        setPlaybackRate(value) {
            this.playbackRate = parseFloat(value);
            if (this.audio) {
                this.audio.playbackRate = this.playbackRate;
            }
        },

        formatTime(seconds) {
            if (isNaN(seconds)) return '0:00';
            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            return `${mins}:${secs.toString().padStart(2, '0')}`;
        },

        // Event handlers
        onLoadedMetadata() {
            if (this.audio) {
                this.duration = this.audio.duration;
                this.isLoading = false;
            }
        },

        onTimeUpdate() {
            if (this.audio) {
                this.currentTime = this.audio.currentTime;
            }
        },

        onEnded() {
            this.isPlaying = false;
            this.currentTime = 0;
        },

        onError() {
            this.error = 'Failed to load audio file';
            this.isLoading = false;
        },

        onCanPlay() {
            this.isLoading = false;
        },

        onWaiting() {
            this.isLoading = true;
        },

        onPlaying() {
            this.isPlaying = true;
            this.isLoading = false;
        },

        onPause() {
            this.isPlaying = false;
        }
    }
}
</script>
</div>
