#!/bin/bash

# PostgreSQL Setup Script for VybePhoto
# This script helps you set up a local PostgreSQL database

set -e

echo "🗄️  VybePhoto PostgreSQL Setup"
echo "================================"
echo ""

# Check if PostgreSQL is installed
if ! command -v psql &> /dev/null; then
    echo "❌ PostgreSQL is not installed."
    echo ""
    echo "Install PostgreSQL:"
    echo "  macOS:    brew install postgresql@16"
    echo "  Ubuntu:   sudo apt install postgresql postgresql-contrib"
    echo "  Windows:  Download from https://www.postgresql.org/download/windows/"
    echo ""
    exit 1
fi

echo "✅ PostgreSQL is installed"
echo ""

# Ask for database details
read -p "Database name [photowebsite]: " DB_NAME
DB_NAME=${DB_NAME:-photowebsite}

read -p "Database user [postgres]: " DB_USER
DB_USER=${DB_USER:-postgres}

read -sp "Database password: " DB_PASSWORD
echo ""

if [ -z "$DB_PASSWORD" ]; then
    echo "❌ Password cannot be empty"
    exit 1
fi

read -p "Database host [localhost]: " DB_HOST
DB_HOST=${DB_HOST:-localhost}

read -p "Database port [5432]: " DB_PORT
DB_PORT=${DB_PORT:-5432}

echo ""
echo "Creating database..."

# Try to create the database
PGPASSWORD=$DB_PASSWORD createdb -h $DB_HOST -p $DB_PORT -U $DB_USER $DB_NAME 2>/dev/null || echo "Database may already exist, continuing..."

echo "✅ Database '$DB_NAME' is ready"
echo ""

# Create or update .env file
ENV_FILE=".env"

if [ -f "$ENV_FILE" ]; then
    echo "⚠️  .env file already exists. Creating backup..."
    cp $ENV_FILE "${ENV_FILE}.backup.$(date +%s)"
fi

cat > $ENV_FILE << EOF
# Database Configuration
USE_POSTGRES=true
DB_HOST=$DB_HOST
DB_PORT=$DB_PORT
DB_USER=$DB_USER
DB_PASSWORD=$DB_PASSWORD
DB_NAME=$DB_NAME

# Application Settings
PORT=3000
NODE_ENV=development

# Admin Authentication (CHANGE THIS!)
ADMIN_PASSWORD=changeme

# Site Configuration
SITE_NAME=VybePhoto
PRIMARY_COLOR=#3b82f6
LOGO_URL=/nrw-web.png

# Age Verification
AGE_VERIFICATION_ENABLED=true

# Rate Limiting
UPLOADS_PER_HOUR=500
EOF

echo "✅ .env file created"
echo ""

echo "🎉 Setup complete!"
echo ""
echo "Next steps:"
echo "  1. Install dependencies:  npm install"
echo "  2. Build CSS:            npm run build:css"
echo "  3. Start the server:     npm start"
echo ""
echo "The database tables will be created automatically on first run."
echo ""
echo "⚠️  IMPORTANT: Change ADMIN_PASSWORD in .env before deploying to production!"

