/* Minimal JS: nav toggle, form enhancement, JSON-LD + GA injection, a11y niceties */
(function () {
  // Mark JS-enabled for progressive enhancement
  document.documentElement.classList.add('js');
  const prefersReduced = window.matchMedia && window.matchMedia('(prefers-reduced-motion: reduce)').matches;

  // Mobile nav toggle
  const toggle = document.getElementById('nav-toggle');
  const mobile = document.getElementById('mobile-nav');
  const overlay = document.getElementById('nav-overlay');
  if (toggle && mobile) {
    toggle.addEventListener('click', () => {
      const expanded = toggle.getAttribute('aria-expanded') === 'true';
      toggle.setAttribute('aria-expanded', String(!expanded));
      mobile.classList.toggle('hidden');
      if (overlay) {
        if (!expanded) {
          overlay.classList.remove('hidden');
          if (!prefersReduced) overlay.animate([{ opacity: 0 }, { opacity: 1 }], { duration: 150 });
        } else {
          overlay.classList.add('hidden');
        }
      }
      if (!prefersReduced) mobile.animate([{ opacity: 0 }, { opacity: 1 }], { duration: 150 });
    });
  }

  // Services dropdown - works with both click and hover
  const servicesDropdown = document.getElementById('services-dropdown');
  const servicesMenu = document.getElementById('services-menu');
  const servicesArrow = document.getElementById('services-arrow');
  
  if (servicesDropdown && servicesMenu && servicesArrow) {
    let timeoutId;
    let isOpen = false;
    
    function showDropdown() {
      clearTimeout(timeoutId);
      isOpen = true;
      servicesMenu.classList.remove('opacity-0', 'invisible', 'hidden');
      servicesMenu.classList.add('opacity-100', 'visible');
      servicesArrow.style.transform = 'rotate(180deg)';
      servicesDropdown.setAttribute('aria-expanded', 'true');
      if (overlay) {
        overlay.classList.remove('hidden');
        if (!prefersReduced) overlay.animate([{ opacity: 0 }, { opacity: 1 }], { duration: 150 });
      }
    }
    
    function hideDropdown() {
      clearTimeout(timeoutId);
      timeoutId = setTimeout(() => {
        isOpen = false;
        servicesMenu.classList.add('opacity-0', 'invisible', 'hidden');
        servicesMenu.classList.remove('opacity-100', 'visible');
        servicesArrow.style.transform = 'rotate(0deg)';
        servicesDropdown.setAttribute('aria-expanded', 'false');
        if (overlay) overlay.classList.add('hidden');
      }, 100);
    }
    
    function toggleDropdown() {
      if (isOpen) {
        hideDropdown();
      } else {
        showDropdown();
      }
    }
    
    // Click to toggle
    servicesDropdown.addEventListener('click', (e) => {
      e.preventDefault();
      toggleDropdown();
    });
    
    // Hover interactions disabled – click only
    
    // Close when clicking outside
    document.addEventListener('click', (e) => {
      if (isOpen && !servicesDropdown.contains(e.target) && !servicesMenu.contains(e.target)) {
        hideDropdown();
      }
    });
    // Overlay click to close
    if (overlay) {
      overlay.addEventListener('click', () => {
        if (isOpen) hideDropdown();
        // Also close mobile menu if open
        if (toggle && mobile && toggle.getAttribute('aria-expanded') === 'true') {
          toggle.setAttribute('aria-expanded', 'false');
          mobile.classList.add('hidden');
          overlay.classList.add('hidden');
        }
      });
    }
    
    // Close on escape key
    document.addEventListener('keydown', (e) => {
      if (e.key === 'Escape' && isOpen) {
        hideDropdown();
        servicesDropdown.focus();
      }
      if (e.key === 'Escape' && toggle && mobile && toggle.getAttribute('aria-expanded') === 'true') {
        toggle.setAttribute('aria-expanded', 'false');
        mobile.classList.add('hidden');
        if (overlay) overlay.classList.add('hidden');
      }
    });
  }

  // Footer year
  const year = document.getElementById('year');
  if (year) year.textContent = String(new Date().getFullYear());

  // Lightweight client-side validation for required fields
  document.querySelectorAll('form[data-validate="true"]').forEach(form => {
    form.addEventListener('submit', (e) => {
      let valid = true;
      form.querySelectorAll('[data-required]')?.forEach((el) => {
        if (!el.value || (el.type === 'checkbox' && !el.checked)) {
          valid = false;
          el.setAttribute('aria-invalid', 'true');
          el.classList.add('ring-2', 'ring-red-500');
        } else {
          el.removeAttribute('aria-invalid');
          el.classList.remove('ring-2', 'ring-red-500');
        }
      });
      if (!valid) {
        e.preventDefault();
        const firstInvalid = form.querySelector('[aria-invalid="true"]');
        firstInvalid && firstInvalid.focus();
      }
    });
  });

  // JSON-LD and GA4 injection from site.config.json (served at site root)
  fetch('/site.config.json').then(r => r.json()).then(cfg => {
    const description = document.querySelector('meta[name="description"]')?.getAttribute('content') || cfg.tagline;
    const sameAs = [cfg.checkatradeUrl, cfg.nextdoorUrl, cfg.facebookUrl, cfg.linkedinUrl].filter(Boolean);
    const ld = {
      '@context': 'https://schema.org',
      '@type': 'ProfessionalService',
      name: cfg.businessName,
      image: '/favicon.ico',
      description,
      url: cfg.siteUrl,
      telephone: cfg.phone,
      email: cfg.email,
      areaServed: cfg.serviceAreas,
      knowsAbout: cfg.primaryKeywords,
      openingHours: cfg.openingHours,
      address: {
        '@type': 'PostalAddress',
        streetAddress: cfg.streetAddress,
        addressLocality: cfg.addressLocality,
        postalCode: cfg.postalCode,
        addressRegion: cfg.region,
        addressCountry: cfg.country
      },
      sameAs
    };
    const script = document.createElement('script');
    script.type = 'application/ld+json';
    script.text = JSON.stringify(ld);
    document.head.appendChild(script);

    if (cfg.gaMeasurementId) {
      const ga = document.createElement('script');
      ga.async = true;
      ga.src = `https://www.googletagmanager.com/gtag/js?id=${cfg.gaMeasurementId}`;
      document.head.appendChild(ga);
      const gtag = document.createElement('script');
      gtag.text = `window.dataLayer=window.dataLayer||[];function gtag(){dataLayer.push(arguments);}gtag('js',new Date());gtag('config','${cfg.gaMeasurementId}');`;
      document.head.appendChild(gtag);
    }
  }).catch(() => {});

  // Reveal on scroll
  if (!prefersReduced && 'IntersectionObserver' in window) {
    const revealEls = document.querySelectorAll('[data-reveal]');
    const io = new IntersectionObserver((entries) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          entry.target.classList.add('is-visible');
          io.unobserve(entry.target);
        }
      });
    }, { rootMargin: '0px 0px -10% 0px', threshold: 0.15 });
    revealEls.forEach(el => io.observe(el));
  } else {
    // If reduced motion is preferred, show immediately
    document.querySelectorAll('[data-reveal]')
      .forEach(el => el.classList.add('is-visible'));
  }

  // SVG line draw on scroll
  if (!prefersReduced) {
    // Vertical wave (single path)
    const wave = document.getElementById('wave-path');
    const waveSvg = document.getElementById('wave-svg');
    if (wave) {
      const total = wave.getTotalLength();
      wave.style.strokeDasharray = String(total);
      wave.style.strokeDashoffset = String(total);
      const drawWave = () => {
        // Scale SVG height to document height so the line spans the page
        if (waveSvg) {
          const docHeight = Math.max(document.documentElement.scrollHeight, document.body.scrollHeight);
          waveSvg.style.height = docHeight + 'px';
        }
        const docH = document.documentElement.scrollHeight - window.innerHeight;
        const prog = docH > 0 ? (window.scrollY || 0) / docH : 0;
        const offset = total * (1 - Math.max(0, Math.min(1, prog)));
        wave.style.strokeDashoffset = String(offset);
      };
      window.addEventListener('scroll', drawWave, { passive: true });
      window.addEventListener('resize', drawWave);
      drawWave();
    }
  }

  // Parallax blobs
  if (!prefersReduced) {
    const blobs = Array.from(document.querySelectorAll('[data-parallax]'));
    const onScrollParallax = () => {
      const y = window.scrollY || 0;
      blobs.forEach((el, i) => {
        const speed = parseFloat(el.getAttribute('data-parallax')) || 0.2;
        el.style.transform = `translateY(${Math.round(y * speed)}px)`;
      });
    };
    window.addEventListener('scroll', onScrollParallax, { passive: true });
    onScrollParallax();
  }

  // Portfolio slide-down detail on index
  (function initPortfolioAccordion(){
    const container = document.querySelector('.container');
    const tiles = document.querySelectorAll('[data-portfolio-id]');
    const detailWrap = document.getElementById('portfolio-detail');
    const detailTitle = document.getElementById('portfolio-detail-title');
    const detailBody = document.getElementById('portfolio-detail-body');
    const btnClose = document.getElementById('portfolio-detail-close');
    if (!tiles.length || !detailWrap || !detailTitle || !detailBody) return;

    const idToTitle = {
      erp: 'Custom ERP For Manufacturing',
      ai: 'Streamlining Tasks With AI',
      finance: 'Sage → Zoho Books Migration',
      workspace: 'Google Workspace Migration & Training',
      itinfra: 'IT Infrastructure Modernisation'
    };

    function openDetail(id){
      const tpl = document.getElementById('tpl-' + id);
      if (!tpl) return;
      detailTitle.textContent = idToTitle[id] || '';
      detailBody.innerHTML = '';
      detailBody.appendChild(tpl.content.cloneNode(true));
      detailWrap.classList.remove('hidden');
      requestAnimationFrame(()=>{
        detailWrap.classList.add('is-open');
      });
      // Scroll panel into view if needed
      const y = detailWrap.getBoundingClientRect().top + window.scrollY - 100;
      window.scrollTo({ top: y, behavior: prefersReduced ? 'auto' : 'smooth' });
    }

    function closeDetail(){
      detailWrap.classList.remove('is-open');
      if (!prefersReduced){
        setTimeout(()=>detailWrap.classList.add('hidden'), 250);
      } else {
        detailWrap.classList.add('hidden');
      }
    }

    tiles.forEach(tile => {
      tile.addEventListener('click', (e)=>{
        const id = tile.getAttribute('data-portfolio-id');
        if (!id) return;
        e.preventDefault();
        // Toggle: if already open with same content, close
        if (!detailWrap.classList.contains('hidden') && detailTitle.textContent === (idToTitle[id]||'')){
          closeDetail();
        } else {
          openDetail(id);
        }
      });
    });

    btnClose && btnClose.addEventListener('click', ()=> closeDetail());
  })();
})();


