/* Minimal JS: nav toggle, form enhancement, JSON-LD + GA injection, a11y niceties */
(function () {
  // Mark JS-enabled for progressive enhancement
  document.documentElement.classList.add('js');
  const prefersReduced = window.matchMedia && window.matchMedia('(prefers-reduced-motion: reduce)').matches;

  // Mobile nav toggle
  const toggle = document.getElementById('nav-toggle');
  const mobile = document.getElementById('mobile-nav');
  if (toggle && mobile) {
    toggle.addEventListener('click', () => {
      const expanded = toggle.getAttribute('aria-expanded') === 'true';
      toggle.setAttribute('aria-expanded', String(!expanded));
      mobile.classList.toggle('hidden');
      if (!prefersReduced) mobile.animate([{ opacity: 0 }, { opacity: 1 }], { duration: 150 });
    });
  }

  // Services dropdown
  const servicesDropdown = document.getElementById('services-dropdown');
  const servicesMenu = document.getElementById('services-menu');
  if (servicesDropdown && servicesMenu) {
    let timeoutId;
    
    servicesDropdown.addEventListener('mouseenter', () => {
      clearTimeout(timeoutId);
      servicesMenu.classList.remove('opacity-0', 'invisible');
      servicesMenu.classList.add('opacity-100', 'visible');
      servicesDropdown.querySelector('svg').style.transform = 'rotate(180deg)';
    });
    
    servicesDropdown.addEventListener('mouseleave', () => {
      timeoutId = setTimeout(() => {
        servicesMenu.classList.add('opacity-0', 'invisible');
        servicesMenu.classList.remove('opacity-100', 'visible');
        servicesDropdown.querySelector('svg').style.transform = 'rotate(0deg)';
      }, 100);
    });
    
    servicesMenu.addEventListener('mouseenter', () => {
      clearTimeout(timeoutId);
    });
    
    servicesMenu.addEventListener('mouseleave', () => {
      timeoutId = setTimeout(() => {
        servicesMenu.classList.add('opacity-0', 'invisible');
        servicesMenu.classList.remove('opacity-100', 'visible');
        servicesDropdown.querySelector('svg').style.transform = 'rotate(0deg)';
      }, 100);
    });
  }

  // Footer year
  const year = document.getElementById('year');
  if (year) year.textContent = String(new Date().getFullYear());

  // Lightweight client-side validation for required fields
  document.querySelectorAll('form[data-validate="true"]').forEach(form => {
    form.addEventListener('submit', (e) => {
      let valid = true;
      form.querySelectorAll('[data-required]')?.forEach((el) => {
        if (!el.value || (el.type === 'checkbox' && !el.checked)) {
          valid = false;
          el.setAttribute('aria-invalid', 'true');
          el.classList.add('ring-2', 'ring-red-500');
        } else {
          el.removeAttribute('aria-invalid');
          el.classList.remove('ring-2', 'ring-red-500');
        }
      });
      if (!valid) {
        e.preventDefault();
        const firstInvalid = form.querySelector('[aria-invalid="true"]');
        firstInvalid && firstInvalid.focus();
      }
    });
  });

  // JSON-LD and GA4 injection from site.config.json (served at site root)
  fetch('/site.config.json').then(r => r.json()).then(cfg => {
    const description = document.querySelector('meta[name="description"]')?.getAttribute('content') || cfg.tagline;
    const sameAs = [cfg.checkatradeUrl, cfg.nextdoorUrl, cfg.facebookUrl, cfg.linkedinUrl].filter(Boolean);
    const ld = {
      '@context': 'https://schema.org',
      '@type': 'ProfessionalService',
      name: cfg.businessName,
      image: '/favicon.ico',
      description,
      url: cfg.siteUrl,
      telephone: cfg.phone,
      email: cfg.email,
      areaServed: cfg.serviceAreas,
      knowsAbout: cfg.primaryKeywords,
      openingHours: cfg.openingHours,
      address: {
        '@type': 'PostalAddress',
        streetAddress: cfg.streetAddress,
        addressLocality: cfg.addressLocality,
        postalCode: cfg.postalCode,
        addressRegion: cfg.region,
        addressCountry: cfg.country
      },
      sameAs
    };
    const script = document.createElement('script');
    script.type = 'application/ld+json';
    script.text = JSON.stringify(ld);
    document.head.appendChild(script);

    if (cfg.gaMeasurementId) {
      const ga = document.createElement('script');
      ga.async = true;
      ga.src = `https://www.googletagmanager.com/gtag/js?id=${cfg.gaMeasurementId}`;
      document.head.appendChild(ga);
      const gtag = document.createElement('script');
      gtag.text = `window.dataLayer=window.dataLayer||[];function gtag(){dataLayer.push(arguments);}gtag('js',new Date());gtag('config','${cfg.gaMeasurementId}');`;
      document.head.appendChild(gtag);
    }
  }).catch(() => {});

  // Reveal on scroll
  if (!prefersReduced && 'IntersectionObserver' in window) {
    const revealEls = document.querySelectorAll('[data-reveal]');
    const io = new IntersectionObserver((entries) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          entry.target.classList.add('is-visible');
          io.unobserve(entry.target);
        }
      });
    }, { rootMargin: '0px 0px -10% 0px', threshold: 0.15 });
    revealEls.forEach(el => io.observe(el));
  } else {
    // If reduced motion is preferred, show immediately
    document.querySelectorAll('[data-reveal]')
      .forEach(el => el.classList.add('is-visible'));
  }

  // SVG line draw on scroll
  if (!prefersReduced) {
    // Vertical wave (single path)
    const wave = document.getElementById('wave-path');
    const waveSvg = document.getElementById('wave-svg');
    if (wave) {
      const total = wave.getTotalLength();
      wave.style.strokeDasharray = String(total);
      wave.style.strokeDashoffset = String(total);
      const drawWave = () => {
        // Scale SVG height to document height so the line spans the page
        if (waveSvg) {
          const docHeight = Math.max(document.documentElement.scrollHeight, document.body.scrollHeight);
          waveSvg.style.height = docHeight + 'px';
        }
        const docH = document.documentElement.scrollHeight - window.innerHeight;
        const prog = docH > 0 ? (window.scrollY || 0) / docH : 0;
        const offset = total * (1 - Math.max(0, Math.min(1, prog)));
        wave.style.strokeDashoffset = String(offset);
      };
      window.addEventListener('scroll', drawWave, { passive: true });
      window.addEventListener('resize', drawWave);
      drawWave();
    }
  }

  // Parallax blobs
  if (!prefersReduced) {
    const blobs = Array.from(document.querySelectorAll('[data-parallax]'));
    const onScrollParallax = () => {
      const y = window.scrollY || 0;
      blobs.forEach((el, i) => {
        const speed = parseFloat(el.getAttribute('data-parallax')) || 0.2;
        el.style.transform = `translateY(${Math.round(y * speed)}px)`;
      });
    };
    window.addEventListener('scroll', onScrollParallax, { passive: true });
    onScrollParallax();
  }
})();


