/* Minimal JS: nav toggle, form enhancement, JSON-LD + GA injection, a11y niceties */
(function () {
  const prefersReduced = window.matchMedia && window.matchMedia('(prefers-reduced-motion: reduce)').matches;

  // Mobile nav toggle
  const toggle = document.getElementById('nav-toggle');
  const mobile = document.getElementById('mobile-nav');
  if (toggle && mobile) {
    toggle.addEventListener('click', () => {
      const expanded = toggle.getAttribute('aria-expanded') === 'true';
      toggle.setAttribute('aria-expanded', String(!expanded));
      mobile.classList.toggle('hidden');
      if (!prefersReduced) mobile.animate([{ opacity: 0 }, { opacity: 1 }], { duration: 150 });
    });
  }

  // Footer year
  const year = document.getElementById('year');
  if (year) year.textContent = String(new Date().getFullYear());

  // Lightweight client-side validation for required fields
  document.querySelectorAll('form[data-validate="true"]').forEach(form => {
    form.addEventListener('submit', (e) => {
      let valid = true;
      form.querySelectorAll('[data-required]')?.forEach((el) => {
        if (!el.value || (el.type === 'checkbox' && !el.checked)) {
          valid = false;
          el.setAttribute('aria-invalid', 'true');
          el.classList.add('ring-2', 'ring-red-500');
        } else {
          el.removeAttribute('aria-invalid');
          el.classList.remove('ring-2', 'ring-red-500');
        }
      });
      if (!valid) {
        e.preventDefault();
        const firstInvalid = form.querySelector('[aria-invalid="true"]');
        firstInvalid && firstInvalid.focus();
      }
    });
  });

  // JSON-LD and GA4 injection from site.config.json (served at site root)
  fetch('/site.config.json').then(r => r.json()).then(cfg => {
    const description = document.querySelector('meta[name="description"]')?.getAttribute('content') || cfg.tagline;
    const sameAs = [cfg.checkatradeUrl, cfg.nextdoorUrl, cfg.facebookUrl, cfg.linkedinUrl].filter(Boolean);
    const ld = {
      '@context': 'https://schema.org',
      '@type': 'ProfessionalService',
      name: cfg.businessName,
      image: '/public/favicon.ico',
      description,
      url: cfg.siteUrl,
      telephone: cfg.phone,
      email: cfg.email,
      areaServed: cfg.serviceAreas,
      knowsAbout: cfg.primaryKeywords,
      openingHours: cfg.openingHours,
      address: {
        '@type': 'PostalAddress',
        streetAddress: cfg.streetAddress,
        addressLocality: cfg.addressLocality,
        postalCode: cfg.postalCode,
        addressRegion: cfg.region,
        addressCountry: cfg.country
      },
      sameAs
    };
    const script = document.createElement('script');
    script.type = 'application/ld+json';
    script.text = JSON.stringify(ld);
    document.head.appendChild(script);

    if (cfg.gaMeasurementId) {
      const ga = document.createElement('script');
      ga.async = true;
      ga.src = `https://www.googletagmanager.com/gtag/js?id=${cfg.gaMeasurementId}`;
      document.head.appendChild(ga);
      const gtag = document.createElement('script');
      gtag.text = `window.dataLayer=window.dataLayer||[];function gtag(){dataLayer.push(arguments);}gtag('js',new Date());gtag('config','${cfg.gaMeasurementId}');`;
      document.head.appendChild(gtag);
    }
  }).catch(() => {});

  // Reveal on scroll
  if (!prefersReduced && 'IntersectionObserver' in window) {
    const revealEls = document.querySelectorAll('[data-reveal]');
    const io = new IntersectionObserver((entries) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          entry.target.classList.add('is-visible');
          io.unobserve(entry.target);
        }
      });
    }, { rootMargin: '0px 0px -10% 0px', threshold: 0.15 });
    revealEls.forEach(el => io.observe(el));
  } else {
    // If reduced motion is preferred, show immediately
    document.querySelectorAll('[data-reveal]')
      .forEach(el => el.classList.add('is-visible'));
  }
})();


