<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Order extends Model
{
    protected $fillable = [
        'order_number',
        'customer_id',
        'status',
        'notes',
        'zoho_invoice_number',
        'currency',
        'totals_estimated_minor',
        'totals_final_minor',
        'placed_at',
        'invoice_sent_at',
        'fulfilled_at',
        'bill_name',
        'bill_line1',
        'bill_line2',
        'bill_city',
        'bill_region',
        'bill_postcode',
        'bill_country_code',
        'bill_phone',
        'ship_name',
        'ship_line1',
        'ship_line2',
        'ship_city',
        'ship_region',
        'ship_postcode',
        'ship_country_code',
        'ship_phone',
    ];

    protected $casts = [
        'totals_estimated_minor' => 'integer',
        'totals_final_minor' => 'integer',
        'placed_at' => 'datetime',
        'invoice_sent_at' => 'datetime',
        'fulfilled_at' => 'datetime',
    ];

    /**
     * Get the customer that owns the order.
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * Get the order lines for the order.
     */
    public function orderLines(): HasMany
    {
        return $this->hasMany(OrderLine::class);
    }

    /**
     * Scope a query to only include orders by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to only include orders by customer.
     */
    public function scopeByCustomer($query, $customerId)
    {
        return $query->where('customer_id', $customerId);
    }

    /**
     * Scope a query to only include placed orders.
     */
    public function scopePlaced($query)
    {
        return $query->whereNotNull('placed_at');
    }

    /**
     * Scope a query to only include fulfilled orders.
     */
    public function scopeFulfilled($query)
    {
        return $query->where('status', 'fulfilled');
    }

    /**
     * Get the estimated total in pounds.
     */
    public function getEstimatedTotalAttribute(): float
    {
        return $this->totals_estimated_minor / 100;
    }

    /**
     * Get the final total in pounds.
     */
    public function getFinalTotalAttribute(): float
    {
        return $this->totals_final_minor / 100;
    }

    /**
     * Check if the order is placed.
     */
    public function isPlaced(): bool
    {
        return !is_null($this->placed_at);
    }

    /**
     * Check if the order is fulfilled.
     */
    public function isFulfilled(): bool
    {
        return $this->status === 'fulfilled';
    }
}
