<?php

namespace App\Services;

use App\Contracts\SafetyService as SafetyServiceContract;
use App\Models\SafetyEvent;
use App\Models\User;
use Illuminate\Support\Str;

class SafetyServiceImpl implements SafetyServiceContract
{
    public function evaluateText(string $text): array
    {
        $normalized = mb_strtolower($text);

        // Minimal heuristic checks; replace with vendor/LLM later.
        if (Str::contains($normalized, ['suicide', 'kill myself', 'self harm'])) {
            return ['action' => 'route_resources', 'rule_code' => 'SELF_HARM_ROUTE', 'excerpt' => $this->scrubExcerpt($text)];
        }

        if (Str::contains($normalized, ['diagnose', 'prescribe', 'medical advice'])) {
            return ['action' => 'redact', 'rule_code' => 'NO_MEDICAL_CLAIMS', 'excerpt' => $this->scrubExcerpt($text)];
        }

        return ['action' => 'allow', 'rule_code' => 'ALLOW', 'excerpt' => null];
    }

    public function checkAndLog(string $context, string $text, ?User $user = null): SafetyEvent
    {
        $result = $this->evaluateText($text);

        return SafetyEvent::create([
            'id' => (string) Str::uuid(),
            'user_id' => $user?->id,
            'context' => $context,
            'rule_code' => $result['rule_code'],
            'action' => $result['action'],
            'excerpt' => $result['excerpt'],
            'created_at' => now(),
        ]);
    }

    private function scrubExcerpt(string $text): string
    {
        $excerpt = trim(Str::limit($text, 160));
        $excerpt = preg_replace('/[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,6}/', '[redacted-email]', $excerpt);
        $excerpt = preg_replace('/\\b\\d{3}[-.\\s]?\\d{3}[-.\\s]?\\d{4}\\b/', '[redacted-phone]', $excerpt);
        return (string) $excerpt;
    }

    public function redact(string $text): string
    {
        return $this->scrubExcerpt($text);
    }
}


