<?php

namespace App\Models;

use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Base model class providing common functionality for all models.
 * 
 * @package App\Models
 */
abstract class BaseModel extends Model
{
    use HasFactory;

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::created(function ($model) {
            $model->clearCache();
        });

        static::updated(function ($model) {
            $model->clearCache();
        });

        static::deleted(function ($model) {
            $model->clearCache();
        });
    }

    /**
     * Scope to get all cached records.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $cacheDuration Cache duration in seconds
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAllCached($query, int $cacheDuration = 3600)
    {
        $cacheKey = 'all_' . get_class($this);
        
        return Cache::remember($cacheKey, $cacheDuration, function () use ($query) {
            return $query->get();
        });
    }

    /**
     * Clear the model's cache.
     * 
     * @return void
     */
    protected function clearCache(): void
    {
        $cacheKey = 'all_' . get_class($this);

        // Check if the cache entry exists before forgetting it
        if (Cache::has($cacheKey)) {
            Cache::forget($cacheKey);
        }
    }
}
