<?php

namespace App\Models;

use App\Casts\Upper;
use App\Casts\TitleCase;
use App\Models\BaseModel;
use App\Models\YarnOrder;
use Illuminate\Support\Facades\Cache;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Suppliers extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;

    public static function boot()
    {
        parent::boot();

        static::saved(function ($supplier) {
            if ($supplier->isDirty('currency')) {
                Cache::tags(['yarn_costs'])->flush();
                Cache::tags(["prices"])->flush();
            }
            self::invalidatePriceResolutions($supplier);
        });

        static::deleted(function ($supplier) {
            self::invalidatePriceResolutions($supplier);
        });

        static::deleting(function($model) {
            $model->shipments?->each?->delete();
            $model->yarns?->each?->delete();
            $model->supplier_addresses?->each?->delete();
            $model->style_versions?->each?->delete();
            $model->yarn_orders?->each?->delete();
        });
        static::restoring(function($model) {
            $model->shipments()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->yarns()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->supplier_addresses()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->style_versions()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
            $model->yarn_orders()->withTrashed()->where('deleted_at', '>=', $model->deleted_at)->each(function ($item, $key) { $item->restore(); });
        });
        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('name');
        });
    }

    protected static function booted()
    {
        static::updated(function ($factory) {
            ShipmentLine::where('factory_id', $factory->id)->update([
                'factory_id' => $factory->id,
            ]);
        });
    }


    protected static $orderByColumn = 'name';

    protected $casts = [
        'name'  => TitleCase::class,
        'contact'  => TitleCase::class,
        'finance_contact'  => TitleCase::class,
        'mid'  => Upper::class,
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    protected $fillable = [
        'type',
        'name',
        'countries_id',
        'contact',
        'email',
        'phone',
        'finance_contact',
        'finance_email',
        'finance_phone',
        'payment_terms',
        'currency',
        'created_at',
        'updated_at',
        'customs',
        'mid',
        'logo',
        'yc_email',
        'deleted_at',
    ];

    public function countries()
    {
        return $this->belongsTo(Countries::class);
    }

    public function shipments()
    {
        return $this->hasMany(Shipment::class);
    }
    public function yarns()
    {
        return $this->hasMany(Yarn::class);
    }
    public function supplier_addresses()
    {
        return $this->hasMany(SupplierAddress::class);
    }
    public function style_versions()
    {
        return $this->hasMany(StyleVersions::class, 'factory_id');
    }
    public function yarn_orders()
    {
        return $this->hasMany(YarnOrder::class);
    }
    public function shipment_lines()
    {
        return $this->hasMany(ShipmentLines::class);
    }

    public function scopeSearch($query, $value){
		$query->where('name', 'like', "%{$value}%")
		->orWhere('id', $value)
		->orWhere('currency', $value)
		->orWhereRelation('countries', 'country', 'like', "%{$value}%")
		->orWhereRelation('countries', 'code', 'like', "%{$value}%");
	}
}
