<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Schema;
use Tests\TestCase;

class DatabaseTest extends TestCase
{
    use RefreshDatabase;

    /** @test */
    public function users_table_exists()
    {
        $this->assertTrue(Schema::hasTable('users'));
    }

    /** @test */
    public function users_table_has_required_columns()
    {
        $this->assertTrue(Schema::hasColumn('users', 'id'));
        $this->assertTrue(Schema::hasColumn('users', 'name'));
        $this->assertTrue(Schema::hasColumn('users', 'email'));
        $this->assertTrue(Schema::hasColumn('users', 'email_verified_at'));
        $this->assertTrue(Schema::hasColumn('users', 'password'));
        $this->assertTrue(Schema::hasColumn('users', 'remember_token'));
        $this->assertTrue(Schema::hasColumn('users', 'created_at'));
        $this->assertTrue(Schema::hasColumn('users', 'updated_at'));
        $this->assertTrue(Schema::hasColumn('users', 'role'));
    }

    /** @test */
    public function sessions_table_exists()
    {
        $this->assertTrue(Schema::hasTable('sessions'));
    }

    /** @test */
    public function sessions_table_has_required_columns()
    {
        $this->assertTrue(Schema::hasColumn('sessions', 'id'));
        $this->assertTrue(Schema::hasColumn('sessions', 'user_id'));
        $this->assertTrue(Schema::hasColumn('sessions', 'ip_address'));
        $this->assertTrue(Schema::hasColumn('sessions', 'user_agent'));
        $this->assertTrue(Schema::hasColumn('sessions', 'payload'));
        $this->assertTrue(Schema::hasColumn('sessions', 'last_activity'));
    }

    /** @test */
    public function password_resets_table_exists()
    {
        $this->assertTrue(Schema::hasTable('password_resets'));
    }

    /** @test */
    public function password_resets_table_has_required_columns()
    {
        $this->assertTrue(Schema::hasColumn('password_resets', 'email'));
        $this->assertTrue(Schema::hasColumn('password_resets', 'token'));
        $this->assertTrue(Schema::hasColumn('password_resets', 'created_at'));
    }

    /** @test */
    public function failed_jobs_table_exists()
    {
        $this->assertTrue(Schema::hasTable('failed_jobs'));
    }

    /** @test */
    public function failed_jobs_table_has_required_columns()
    {
        $this->assertTrue(Schema::hasColumn('failed_jobs', 'id'));
        $this->assertTrue(Schema::hasColumn('failed_jobs', 'uuid'));
        $this->assertTrue(Schema::hasColumn('failed_jobs', 'connection'));
        $this->assertTrue(Schema::hasColumn('failed_jobs', 'queue'));
        $this->assertTrue(Schema::hasColumn('failed_jobs', 'payload'));
        $this->assertTrue(Schema::hasColumn('failed_jobs', 'exception'));
        $this->assertTrue(Schema::hasColumn('failed_jobs', 'failed_at'));
    }

    /** @test */
    public function personal_access_tokens_table_exists()
    {
        $this->assertTrue(Schema::hasTable('personal_access_tokens'));
    }

    /** @test */
    public function personal_access_tokens_table_has_required_columns()
    {
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'id'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'tokenable_type'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'tokenable_id'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'name'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'token'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'abilities'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'last_used_at'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'expires_at'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'created_at'));
        $this->assertTrue(Schema::hasColumn('personal_access_tokens', 'updated_at'));
    }

    /** @test */
    public function can_create_user_with_role()
    {
        $user = User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        $this->assertDatabaseHas('users', [
            'name' => 'Test User',
            'email' => 'test@example.com',
            'role' => 'admin',
        ]);

        $this->assertEquals('admin', $user->role);
    }

    /** @test */
    public function user_role_defaults_to_user()
    {
        $user = User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            // role not specified
        ]);

        $this->assertEquals('user', $user->role);
    }

    /** @test */
    public function can_update_user_role()
    {
        $user = User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        $user->update(['role' => 'admin']);

        $this->assertEquals('admin', $user->fresh()->role);
        $this->assertDatabaseHas('users', [
            'email' => 'test@example.com',
            'role' => 'admin',
        ]);
    }

    /** @test */
    public function database_connection_is_sqlite_for_testing()
    {
        $connection = config('database.default');
        $this->assertEquals('sqlite', $connection);
    }

    /** @test */
    public function can_query_users_by_role()
    {
        User::create([
            'name' => 'Admin User',
            'email' => 'admin@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        User::create([
            'name' => 'Regular User',
            'email' => 'user@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        $admins = User::where('role', 'admin')->get();
        $users = User::where('role', 'user')->get();

        $this->assertCount(1, $admins);
        $this->assertCount(1, $users);
        $this->assertEquals('admin@example.com', $admins->first()->email);
        $this->assertEquals('user@example.com', $users->first()->email);
    }

    /** @test */
    public function can_use_user_model_scopes()
    {
        User::create([
            'name' => 'Admin User',
            'email' => 'admin@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        User::create([
            'name' => 'Regular User',
            'email' => 'user@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        // Test that we can filter by role using model methods
        $adminUsers = User::all()->filter(function ($user) {
            return $user->isAdmin();
        });

        $regularUsers = User::all()->filter(function ($user) {
            return $user->isUser();
        });

        $this->assertCount(1, $adminUsers);
        $this->assertCount(1, $regularUsers);
    }

    /** @test */
    public function user_email_is_unique()
    {
        User::create([
            'name' => 'First User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        $this->expectException(\Illuminate\Database\QueryException::class);

        User::create([
            'name' => 'Second User',
            'email' => 'test@example.com', // Same email
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);
    }

    /** @test */
    public function user_password_is_hashed()
    {
        $user = User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        $this->assertNotEquals('password', $user->password);
        $this->assertTrue(Hash::check('password', $user->password));
    }

    /** @test */
    public function can_soft_delete_users()
    {
        $user = User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        $user->delete();

        $this->assertSoftDeleted('users', [
            'email' => 'test@example.com',
        ]);
    }
}