#!/usr/bin/env node

const { Pool } = require('pg');
const bcrypt = require('bcrypt');
const fs = require('fs');

require('dotenv').config();

// Database configuration
const dbConfig = process.env.INSTANCE_UNIX_SOCKET
  ? {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.INSTANCE_UNIX_SOCKET,
    }
  : {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || '5432', 10),
    };

const pool = new Pool(dbConfig);

// Default superuser credentials
const DEFAULT_USERNAME = 'admin';
const DEFAULT_EMAIL = 'admin@vybephoto.com';
const DEFAULT_PASSWORD = 'admin123456'; // Change this after first login!

async function runMigration() {
  console.log('Step 1: Running superuser migration...\n');
  
  const migrationPath = './migrations/002-add-superuser-support.sql';
  
  if (!fs.existsSync(migrationPath)) {
    console.log('Warning: Migration file not found at', migrationPath);
    console.log('The database may already have superuser support\n');
    return;
  }
  
  const sql = fs.readFileSync(migrationPath, 'utf8');
  
  try {
    await pool.query(sql);
    console.log('✓ Migration completed successfully\n');
  } catch (error) {
    if (error.message.includes('already exists')) {
      console.log('✓ Tables already exist, skipping migration\n');
    } else {
      console.error('✗ Migration failed:', error.message);
      throw error;
    }
  }
}

async function createSuperuser() {
  console.log('Step 2: Creating superuser account...\n');
  
  const username = DEFAULT_USERNAME;
  const email = DEFAULT_EMAIL;
  const password = DEFAULT_PASSWORD;
  
  console.log('Using default credentials:');
  console.log('  Username:', username);
  console.log('  Email:', email);
  console.log('  Password:', password);
  console.log('  ⚠️  CHANGE THE PASSWORD AFTER FIRST LOGIN!\n');
  
  console.log('Generating password hash...');
  const passwordHash = await bcrypt.hash(password, 10);
  
  console.log('Creating superuser account...');
  
  try {
    const result = await pool.query(
      `INSERT INTO users (
        username, 
        email, 
        password_hash, 
        full_name,
        subscription_tier, 
        subscription_status,
        is_superuser,
        max_photos,
        max_storage_bytes
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
      ON CONFLICT (username) DO UPDATE SET
        is_superuser = TRUE,
        subscription_status = 'active',
        subscription_tier = 'business'
      RETURNING id, username, email, is_superuser`,
      [
        username.toLowerCase().trim(),
        email.toLowerCase().trim(),
        passwordHash,
        'Superuser Administrator',
        'business',
        'active',
        true,
        999999,
        999999999999
      ]
    );
    
    console.log('\n===================================');
    console.log('✓ Superuser created successfully!');
    console.log('===================================\n');
    console.log('User ID:', result.rows[0].id);
    console.log('Username:', result.rows[0].username);
    console.log('Email:', result.rows[0].email);
    console.log('Superuser:', result.rows[0].is_superuser);
    
  } catch (error) {
    if (error.code === '23505') { // Duplicate key - email conflict
      console.log('\nEmail already exists. Updating existing user to superuser...\n');
      
      const result = await pool.query(
        `UPDATE users 
         SET is_superuser = TRUE,
             subscription_status = 'active',
             subscription_tier = 'business'
         WHERE email = $1
         RETURNING id, username, email, is_superuser`,
        [email.toLowerCase().trim()]
      );
      
      if (result.rows.length > 0) {
        console.log('✓ Existing user updated to superuser status');
        console.log('User ID:', result.rows[0].id);
        console.log('Username:', result.rows[0].username);
        console.log('Email:', result.rows[0].email);
      } else {
        throw new Error('Failed to update user');
      }
    } else {
      throw error;
    }
  }
  
  console.log('\n===================================');
  console.log('Login Information:');
  console.log('===================================');
  console.log('URL: http://app.localhost:3000/login');
  console.log('Username:', username);
  console.log('Password:', password);
  console.log('\nSuperuser Portal: http://app.localhost:3000/superuser');
  console.log('\n⚠️  IMPORTANT: Change the password after first login!\n');
}

async function main() {
  console.log('===================================');
  console.log('Vybe Photo - Superuser Setup');
  console.log('===================================\n');
  
  console.log('Database Configuration:');
  console.log('  Host:', dbConfig.host);
  console.log('  Port:', dbConfig.port || 'unix socket');
  console.log('  Database:', dbConfig.database);
  console.log('  User:', dbConfig.user);
  console.log('');
  
  try {
    // Test connection
    await pool.query('SELECT 1');
    console.log('✓ Database connection successful\n');
    
    await runMigration();
    await createSuperuser();
    
    console.log('Setup complete!');
    
  } catch (error) {
    console.error('\n✗ Error:', error.message);
    console.error(error);
    process.exit(1);
  } finally {
    await pool.end();
  }
}

main();


