<?php

namespace App\Console\Commands;

use App\Models\Colourways;
use App\Services\PriceResolutionService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class BackfillPriceResolutions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'price-resolutions:backfill 
                            {--chunk=100 : Number of colourways to process at once}
                            {--dry-run : Show what would be processed without actually doing it}
                            {--force : Skip confirmation prompt}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Backfill price resolutions cache for existing colourways';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $chunkSize = (int) $this->option('chunk');
        $dryRun = $this->option('dry-run');
        $force = $this->option('force');

        $this->info('Price Resolution Backfill Command');
        $this->info('================================');

        // Get total count of colourways to process
        $totalColourways = Colourways::count();
        $this->info("Total colourways to process: {$totalColourways}");

        if ($dryRun) {
            $this->warn('DRY RUN MODE - No data will be written');
        }

        if (!$force && !$dryRun) {
            if (!$this->confirm('Do you want to proceed with backfilling price resolutions?')) {
                $this->info('Operation cancelled.');
                return 0;
            }
        }

        $service = app(PriceResolutionService::class);
        $processed = 0;
        $errors = 0;
        $resolutionsCreated = 0;

        $this->info("Processing colourways in chunks of {$chunkSize}...");
        $this->newLine();

        // Process colourways in chunks
        Colourways::with([
            'style_versions.styles.seasons',
            'style_versions.factories',
            'style_versions.styles.customers'
        ])->chunk($chunkSize, function ($colourways) use ($service, &$processed, &$errors, &$resolutionsCreated, $dryRun) {
            foreach ($colourways as $colourway) {
                $processed++;
                
                if ($dryRun) {
                    $this->line("Would process colourway {$colourway->id} ({$colourway->name})");
                    continue;
                }

                try {
                    // Get all sizes and phases that might need resolution
                    $sizes = DB::table('sizes')->pluck('id');
                    $phases = DB::table('phases')->pluck('id');
                    $seasonId = $colourway->style_versions->styles->seasons_id;
                    $styleVersionId = $colourway->style_versions_id;

                    $colourwayResolutions = 0;

                    // Compute resolution for each combination
                    foreach ($sizes as $sizeId) {
                        foreach ($phases as $phaseId) {
                            $resolution = $service->resolve($styleVersionId, $colourway->id, $sizeId, $phaseId, $seasonId);
                            if ($resolution) {
                                $colourwayResolutions++;
                            }
                        }
                    }

                    // Also compute for null size/phase combinations
                    $resolution = $service->resolve($styleVersionId, $colourway->id, null, null, $seasonId);
                    if ($resolution) {
                        $colourwayResolutions++;
                    }

                    $resolutionsCreated += $colourwayResolutions;

                    if ($processed % 10 === 0) {
                        $this->line("Processed {$processed} colourways, created {$resolutionsCreated} resolutions");
                    }

                } catch (\Exception $e) {
                    $errors++;
                    $this->error("Error processing colourway {$colourway->id}: " . $e->getMessage());
                }
            }
        });

        $this->newLine();
        $this->info('Backfill completed!');
        $this->info("Total colourways processed: {$processed}");
        $this->info("Total resolutions created: {$resolutionsCreated}");
        
        if ($errors > 0) {
            $this->warn("Errors encountered: {$errors}");
        }

        // Show cache statistics
        if (!$dryRun) {
            $stats = $service->getCacheStats();
            $this->newLine();
            $this->info('Cache Statistics:');
            $this->info("Total resolutions: {$stats['total']}");
            $this->info("Fresh resolutions: {$stats['fresh']}");
            $this->info("Stale resolutions: {$stats['stale']}");
            $this->info("Fresh percentage: {$stats['fresh_percentage']}%");
        }

        return 0;
    }
}
