<?php

namespace App\Http\Livewire\Sales\Pricing;

use App\Models\Price;
use App\Models\Sizes;
use App\Models\Styles;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Helper\Functions;
use App\Models\Suppliers;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use App\Models\StyleVersions;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;
use App\Models\CustomerOrderLineQuantities;

class PricingSheetDev extends FilterableComponent
{
	use WithPagination;
    use Filterable;

    public $hideFilters = false, $selected;

    #[Validate([
        'season' => ['required', 'exists:seasons,id'], // Assuming seasons start from 1
        'department' => ['nullable', 'exists:departments,id'], // Adjust table name if needed
        'customer' => ['nullable', 'exists:customers,id'],
        'factory' => ['nullable', 'exists:suppliers,id'],
        'coo' => ['nullable', 'exists:countries,id'], // Assuming 'coo' refers to country of origin
        'category' => ['nullable', 'in:ladies,mens,childrens,accessories'], // Adjust table name if needed
        'search' => ['nullable', 'string', 'max:30'],
        'cancelled' => ['nullable', 'boolean'], // Assuming it's a true/false flag
    ])]
    public
        $season = 1,
        $department,
        $customer = 1,
        $factory,
        $coo,
        $category,
        $search,
        $cancelled;
    public
        $goto;

    protected function filterKey(): string
    {
        return 'whpricingfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.1;
    }

    public $columns = [
        'Style' => TRUE,
        'Style Version' => TRUE,
        'Image' => TRUE,
        'Yarns' => TRUE,
        'Price' => TRUE,
        'Apply To' => TRUE,
        'Colour Type' => TRUE,
        'Colourway' => TRUE,
        'Yarn/Kg' => TRUE,
        'Yarn Transport' => TRUE,
        'Total Yarn Cost' => TRUE,
        'Sample Weight' => TRUE,
        'Gross Weight' => TRUE,
        'Yarn Value' => TRUE,
        'CMT' => TRUE,
        'CMT History' => TRUE,
        'Valid Till' => TRUE,
        'AOC (%)' => TRUE,
        'Accessories' => TRUE,
        'Labels' => TRUE,
        'Gmt Transport' => TRUE,
        'Testing' => TRUE,
        'Testing Courier' => TRUE,
        'Embroidery' => TRUE,
        'Duty' => TRUE,
        'Subtotal' => TRUE,
        'With Customs' => TRUE,
        'Subtotal £' => TRUE,
        'Quote' => TRUE,
        'Quotes Sent' => TRUE,
        'Quote History' => TRUE,
        'Target' => TRUE,
        'MOQ' => TRUE,
        'Internal Notes' => TRUE,
        'Customer Notes' => TRUE,
        'Share' => TRUE,
        'Last Update' => TRUE,
    ];

    protected function filters(): array
    {
        return ['season', 'department', 'customer', 'factory', 'coo', 'category', 'search', 'cancelled', 'columns'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['season', 'department', 'customer', 'factory', 'coo', 'category', 'search', 'cancelled'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
            $this->loadPrices();
        }
    }

    public $styles = [];

    public function mount()
    {
        $this->loadFilterSettings();
        $this->loadPrices();
    }

    public function render()
    {
        return view('livewire.sales.pricing.pricing-sheet-dev');
    }

    public function getPriceInStyleCount(Styles $style){
        return $style->style_versions->flatMap(function ($version) {
            return $version->prices;
        })->count();
    }
    public function getPriceInVersionCount(StyleVersions $version){
        return $version->prices->count();
    }

    #[Computed(cache: true, key: 'factories-with-countries')]
    public function factories(){
        return Suppliers::with('countries')->where('type', 'factory')->get();
    }
    #[Computed]
    public function sizes(){
        return Sizes::get();
    }

    #[Computed]
    public function seasons(){
        return Seasons::allCached();
    }

    #[Computed]
    public function customers(){
        return Customer::allCached();
    }

    #[Computed]
    public function departments(){
        return Departments::allCached();
    }

    #[On('toggle-selected-price')]
    public function toggleSelected($id){
        if(empty($this->selected[$id])){
            $this->selected[$id] = TRUE;
        }
        else{
            unset($this->selected[$id]);
        }
    }

    public function customerPrint(){
        $list = array_keys($this->selected);
        return redirect()->route('pricingcustomerprint', json_encode($list));
    }

    public function refreshAllPriceCaches(){
        $colq = CustomerOrderLineQuantities::
        with('customer_order_lines.colourways.style_versions.styles.seasons', 'customer_order_lines.colourways.style_versions.prices')->
        whereRelation('customer_order_lines.customer_orders', 'order_type', 'wholesale')->
        get();
        // dd($colq->pluck('id'), $this->styles[$p]['style_version_id']);

        foreach($colq as $line){
            $price = Functions::getPrice(
                $line->customer_order_lines->colourways->style_versions->prices,
                $line->sizes_id,
                $line->customer_order_lines->colourways->colour_type,
                $line->customer_order_lines->colourways->id
            );

            // dd($price, $line->sizes_id, $line->customer_order_lines->colourways->style_versions->prices);

            if(!is_string($price)){
                // dd($price);
                $line->update([
                    'prices_id' => $price->id,
                    // Price resolution system will handle caching automatically
                    // No need to set old cache columns
                ]);
            }
        }

        dd('done');
    }
    public function loadPrices(){

        if(empty($this->season))
            $this->season = 1;
        if(empty($this->customer))
            $this->customer = 1;

        $where = [];
        $where[] = "\nwhere 1 ";
        $where[] = "\nand style_versions.deleted_at IS NULL ";
        if(!empty($this->goto)){
            $where[] = "and styles.id = {$this->goto}";
        }
        else{
            $where[] = "and styles.customers_id = {$this->customer}";
            $where[] = "and styles.seasons_id = {$this->season}";
            // if(!$this->cancelled)
            //     $where[] = "and styles.cancelled != 1";
            if(!empty($this->factory))
                $where[] = "and suppliers.id = {$this->factory}";
            if(!empty($this->department))
                $where[] = "and styles.departments_id = {$this->department}";
            if(!empty($this->coo))
                $where[] = "and suppliers.countries_id = {$this->coo}";
            if(!empty($this->category))
                $where[] = "and styles.category = '{$this->category}'";
            if(!empty($this->search)){
                $where[] = "\nAND (
                    designs.description like '%{$this->search}%'
                    or designs.id like '%{$this->search}%'
                    or styles.customer_ref like '%{$this->search}%'
                    or style_versions.factory_id like '%{$this->search}%'
                    or prices.name like '%{$this->search}%'
                    or style_versions.name like '%{$this->search}%'
                    or colourways.name like '%{$this->search}%'
                    or prices.quote like '%{$this->search}%'
                    or prices.cmt like '%{$this->search}%'
                    )";
            }
        }

        $query = "
            select
                prices.id,
                prices.yarn AS yarnkg,
                prices.yarn_currency AS yarnkg_currency,
                designs.id AS design_id,
                designs.description AS description,
                styles.id AS style_id,
                styles.customer_ref AS customer_ref,
                styles.seasons_id AS seasons_id,
                seasons.euro_rate AS euro_rate,
                seasons.usd_rate AS usd_rate,
                style_versions.id AS style_version_id,
                style_versions.name AS style_version_name,
                suppliers.id AS factory_id,
                suppliers.name AS factory,
                prices.id AS price_id,
                prices.name AS price_name,
                prices.model AS model,
                prices.comments_toggle AS comments_toggle,
                prices.sizes_id AS sizes_id,
                prices.colour_type AS colour_type,
                prices.colourways_id AS colourways_id,
                prices.yarn_trans_currency AS yarn_trans_currency,
                format(prices.yarn_trans, 2) AS yarn_trans,
                format(prices.weight, 3) AS weight,
                format(prices.costed_weight, 3) AS costed_weight,
                format(prices.cmt, 2) AS cmt,
                prices.cmt_status AS cmt_status,
                prices.valid_until AS valid_until,
                prices.aoc AS aoc,
                customers.discount as cust_discount,
                prices.accessory_currency AS accessory_currency,
                format(prices.accessory, 2) AS accessory,
                prices.label_currency AS label_currency,
                format(prices.label, 2) AS label,
                prices.gmt_trans_currency AS gmt_trans_currency,
                format(prices.gmt_trans, 2) AS gmt_trans,
                prices.embroidery_currency AS embroidery_currency,
                format(prices.embroidery, 2) AS embroidery,
                format(prices.testing, 2) AS testing,
                format(prices.testing_courier, 2) AS testing_courier,
                prices.quote_status AS quote_status,
                format(prices.target, 2) AS target,
                prices.min_qty AS min_qty,
                prices.duty,
                prices.notes AS notes,
                prices.customer_notes AS customer_notes,
                suppliers.currency AS fact_currency,
                customers.currency AS cust_currency,
                customers.id AS customer_id,
                customers.customs AS customer_customs,
                suppliers.customs AS factory_customs,
                seasons.locked AS season_locked,
                styles.intake_id,
                intakes.description as intake,
                (
                    SELECT
                        JSON_ARRAYAGG(
                            JSON_OBJECT(
                                'percentage', percentage,
                                'days', days
                            )
                        )
                    FROM
                        customer_payment_terms
                    WHERE
                        customer_payment_terms.customer_id = customers.id
                ) AS cust_terms,
                (
                    select
                        c.image
                    from
                            style_versions sv
                            left join colourways c on sv.id = style_versions.id
                    where c.style_versions_id = sv.id and (c.image is not null)
                    limit 1
                ) AS image,
                (
                    select
                        case when count(0) = 0
                        then 1
                        else count(0)
                        end
                    from
                        prices p
                    where p.style_versions_id = style_versions.id
                ) AS no_style_version_prices,
                COALESCE((
                    select
                        case when count(p.id) = 0
                        then 0
                        else 1
                        end
                    from
                        prices p
                    where p.style_versions_id = style_versions.id
                ), 0) AS got_prices,
                SUM(prices.id) OVER (PARTITION BY style_versions.id) as prices_count,
                (
                    select
                        group_concat(
                            distinct y.description separator ','
                        )
                    from yarns y
                    join yarn_colours yc on yc.yarn_id = y.id
                    join colourway_yarns cy on cy.yarn_colours_id = yc.id
                    join colourways c on c.id = cy.colourways_id
                    join style_versions sv on sv.id = c.style_versions_id
                    where sv.id = style_versions.id
                ) AS yarn,
                (
                    select
                        concat(
                            '[',
                            group_concat(
                                json_object(
                                    'name', c.name, 'id', c.id
                                ) separator ','
                            ),
                            ']'
                        )
                    from
                        colourways c
                    where c.style_versions_id = style_versions.id
                ) AS colourways,





                (
                    SELECT
                        CONCAT('[', GROUP_CONCAT(
                            JSON_OBJECT(
                                'season', prev_season.description,
                                'price_id', prev_p.id,
                                'cmt', prev_p.cmt,
                                'quote', prev_p.quote
                            )
                            ORDER BY prev_season.id DESC
                            SEPARATOR ','
                        ), ']')
                    FROM prices AS prev_p
                    JOIN style_versions AS prev_sv ON prev_p.style_versions_id = prev_sv.id
                    JOIN styles AS prev_s ON prev_sv.styles_id = prev_s.id
                    JOIN seasons AS prev_season ON prev_season.id=prev_s.seasons_id
                    WHERE prev_s.designs_id = styles.designs_id
                        AND prev_s.seasons_id < seasons.id
                        AND prev_p.sizes_id = prices.sizes_id
                        AND prev_p.quote_status = 'confirmed'
                    ORDER BY prev_season.id DESC
                    LIMIT 3
                ) AS historic,



                prices.quotes_sent AS quotes_sent,
                prices.weight_comments AS weight_comments,
                prices.cost_comments AS cost_comments,
                format(prices.quote, 2) AS quote,
                convert_currency(prices.quote, customers.currency, '£', styles.seasons_id) AS quote_gbp,
                suppliers.countries_id AS coo,
                countries.country AS coo_name,
                styles.departments_id AS departments_id,
                styles.category AS category,
                styles.cancelled AS style_cancelled,
                sizes.name AS sizes_name,
                colourways.name AS colourways_name,
                (
                    select users.name
                    from audits
                    join users on users.id = audits.user_id
                    where audits.auditable_type = 'App\\\Models\\\Price'
                    and audits.auditable_id = prices.id
                    order by audits.created_at desc
                    limit 1
                ) AS last_updated_by,






                0 AS yarn_cost_kg_eur -- will be computed in PHP using the colourways relationship


            from styles
            join seasons on seasons.id = styles.seasons_id
            join designs on designs.id = styles.designs_id
            join style_versions on style_versions.styles_id = styles.id
            join suppliers on suppliers.id = style_versions.factory_id
            join countries on countries.id = suppliers.countries_id
            join customers on customers.id = styles.customers_id
            left join intakes on intakes.id = styles.intake_id
            left join prices on prices.style_versions_id = style_versions.id and prices.deleted_at IS NULL
            left join sizes on sizes.id = prices.sizes_id
            left join colourways on colourways.id = prices.colourways_id ";
            foreach($where as $wr){
                $query = $query . $wr . " \n";
            }
            $query = $query . "
            order by
            styles.intake_id,
            style_versions.styles_id,
            style_versions.id,
            sizes.order
        ";

        $this->styles = collect(DB::select($query));

        // Add yarn_cost_kg_eur for each style by loading the colourway relationship
        $this->styles = $this->styles->map(function ($style) {
            if (isset($style->colourways_id) && $style->colourways_id) {
                $colourway = \App\Models\Colourways::with('colourway_yarns.yarn_colours.yarn.suppliers', 'style_versions.styles.seasons')
                    ->find($style->colourways_id);
                $style->yarn_cost_kg_eur = $colourway ? $colourway->yarn_cost_kg_eur : 0;
            } else {
                $style->yarn_cost_kg_eur = 0;
            }
            return $style;
        });

        $this->styles = $this->styles->groupBy(['intake', 'style_id', 'style_version_id']);

        // dd($this->styles->first()->first()->first()->last_updated_by);
    }
}
