<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Services\CustomSessionService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class CustomSessionController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {}

    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'topic' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'duration_minutes' => 'required|integer|min:5|max:120',
            'voice_preference' => 'nullable|string|in:male,female,neutral',
            'special_notes' => 'nullable|string|max:1000',
            'goals' => 'nullable|array',
            'goals.*' => 'string|max:255',
        ]);

        $user = Auth::user();
        $customRequest = $this->customSessionService->createRequest($request->all(), $user);

        return response()->json([
            'data' => [
                'id' => $customRequest->id,
                'topic' => $customRequest->topic,
                'description' => $customRequest->description,
                'durationMinutes' => $customRequest->duration_minutes,
                'voicePreference' => $customRequest->voice_preference,
                'specialNotes' => $customRequest->special_notes,
                'goals' => $customRequest->goals,
                'status' => $customRequest->status,
                'createdAt' => $customRequest->created_at,
            ],
            'message' => 'Custom session request created successfully',
        ], 201);
    }

    public function index(Request $request): JsonResponse
    {
        $user = Auth::user();
        $status = $request->get('status');
        
        $sessions = $this->customSessionService->getUserSessions($user, $status);

        return response()->json([
            'data' => $sessions->map(function ($session) {
                return [
                    'id' => $session->id,
                    'title' => $session->title,
                    'description' => $session->description,
                    'status' => $session->status,
                    'durationSeconds' => $session->duration_seconds,
                    'formattedDuration' => $session->formatted_duration,
                    'generatedAt' => $session->generated_at,
                    'customRequest' => [
                        'id' => $session->customRequest->id,
                        'topic' => $session->customRequest->topic,
                        'durationMinutes' => $session->customRequest->duration_minutes,
                        'voicePreference' => $session->customRequest->voice_preference,
                    ],
                    'createdAt' => $session->created_at,
                    'updatedAt' => $session->updated_at,
                ];
            }),
        ]);
    }

    public function show(int $id): JsonResponse
    {
        $user = Auth::user();
        $session = $this->customSessionService->getSessionById($id, $user);

        if (!$session) {
            return response()->json([
                'error' => 'Custom session not found',
            ], 404);
        }

        return response()->json([
            'data' => [
                'id' => $session->id,
                'title' => $session->title,
                'description' => $session->description,
                'transcript' => $session->transcript,
                'status' => $session->status,
                'durationSeconds' => $session->duration_seconds,
                'formattedDuration' => $session->formatted_duration,
                'audioPath' => $session->audio_path,
                'imagePath' => $session->image_path,
                'generatedAt' => $session->generated_at,
                'generationMetadata' => $session->generation_metadata,
                'customRequest' => [
                    'id' => $session->customRequest->id,
                    'topic' => $session->customRequest->topic,
                    'description' => $session->customRequest->description,
                    'durationMinutes' => $session->customRequest->duration_minutes,
                    'voicePreference' => $session->customRequest->voice_preference,
                    'specialNotes' => $session->customRequest->special_notes,
                    'goals' => $session->customRequest->goals,
                ],
                'createdAt' => $session->created_at,
                'updatedAt' => $session->updated_at,
            ],
        ]);
    }

    public function audioUrl(int $id): JsonResponse
    {
        $user = Auth::user();
        $session = $this->customSessionService->getSessionById($id, $user);

        if (!$session) {
            return response()->json([
                'error' => 'Custom session not found',
            ], 404);
        }

        if (!$session->isReady()) {
            return response()->json([
                'error' => 'Audio not ready yet',
            ], 400);
        }

        $audioUrl = $this->customSessionService->getSignedAudioUrl($session);

        return response()->json([
            'data' => [
                'audioUrl' => $audioUrl,
                'expiresAt' => now()->addMinutes(30)->toISOString(),
            ],
        ]);
    }

    public function imageUrl(int $id): JsonResponse
    {
        $user = Auth::user();
        $session = $this->customSessionService->getSessionById($id, $user);

        if (!$session) {
            return response()->json([
                'error' => 'Custom session not found',
            ], 404);
        }

        if (!$session->image_path) {
            return response()->json([
                'error' => 'No image available',
            ], 404);
        }

        $imageUrl = $this->customSessionService->getSignedImageUrl($session);

        return response()->json([
            'data' => [
                'imageUrl' => $imageUrl,
                'expiresAt' => now()->addMinutes(30)->toISOString(),
            ],
        ]);
    }
}
