<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Services\CustomSessionService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;

class CustomSessionController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {}

    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'topic' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'length' => 'sometimes|string|in:5min,15min,30min,45min,60min',
            'voicePreference' => 'sometimes|string|in:female,male,neutral',
            'specialNotes' => 'sometimes|nullable|string|max:1000',
        ]);
        
        $user = Auth::user();
        
        $customRequest = $this->customSessionService->createRequest([
            'topic' => $request->topic,
            'description' => $request->description,
            'length' => $request->length,
            'voice_preference' => $request->voicePreference,
            'special_notes' => $request->specialNotes,
        ], $user);
        
        // Process the request immediately for demo purposes
        $this->customSessionService->processCustomRequest($customRequest);
        
        return response()->json([
            'message' => 'Custom session request created successfully',
            'data' => [
                'id' => $customRequest->id,
                'topic' => $customRequest->topic,
                'status' => $customRequest->status,
                'createdAt' => $customRequest->created_at,
            ],
        ], Response::HTTP_CREATED);
    }

    public function index(Request $request): JsonResponse
    {
        $user = Auth::user();
        $filters = $request->only(['status']);
        
        $sessions = $this->customSessionService->getUserSessions($user, $filters);
        
        return response()->json([
            'data' => $sessions->items(),
            'meta' => [
                'current_page' => $sessions->currentPage(),
                'last_page' => $sessions->lastPage(),
                'per_page' => $sessions->perPage(),
                'total' => $sessions->total(),
            ],
        ]);
    }

    public function show(int $id): JsonResponse
    {
        $user = Auth::user();
        $session = $this->customSessionService->getSession($id, $user);
        
        if (!$session) {
            return response()->json([
                'error' => 'Session not found',
                'message' => 'The requested custom session could not be found.',
            ], Response::HTTP_NOT_FOUND);
        }
        
        return response()->json([
            'data' => [
                'id' => $session->id,
                'title' => $session->title,
                'description' => $session->description,
                'audioUrl' => $this->customSessionService->getSessionAudioUrl($session),
                'transcript' => $session->transcript,
                'duration' => $session->duration,
                'status' => $session->status,
                'generatedAt' => $session->generated_at,
                'generationMetadata' => $session->generation_metadata,
                'customRequest' => [
                    'id' => $session->customRequest->id,
                    'topic' => $session->customRequest->topic,
                    'length' => $session->customRequest->length,
                    'voicePreference' => $session->customRequest->voice_preference,
                ],
                'createdAt' => $session->created_at,
                'updatedAt' => $session->updated_at,
            ],
        ]);
    }

    public function requests(Request $request): JsonResponse
    {
        $user = Auth::user();
        $filters = $request->only(['status']);
        
        $requests = $this->customSessionService->getUserRequests($user, $filters);
        
        return response()->json([
            'data' => $requests->items(),
            'meta' => [
                'current_page' => $requests->currentPage(),
                'last_page' => $requests->lastPage(),
                'per_page' => $requests->perPage(),
                'total' => $requests->total(),
            ],
        ]);
    }
}
