<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Services\CustomSessionService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class CustomSessionController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {}

    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'topic' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'sessionLength' => 'required|integer|min:300|max:3600', // 5 min to 1 hour
            'voicePreference' => 'required|string|in:male,female,neutral',
            'specialNotes' => 'nullable|string|max:1000',
        ]);

        $customRequest = CustomRequest::create([
            'user_id' => Auth::id(),
            'topic' => $request->topic,
            'description' => $request->description,
            'session_length' => $request->sessionLength,
            'voice_preference' => $request->voicePreference,
            'special_notes' => $request->specialNotes,
            'status' => 'pending',
        ]);

        // Queue the session generation job
        $this->customSessionService->queueSessionGeneration($customRequest);

        return response()->json([
            'message' => 'Custom session request submitted successfully',
            'request' => [
                'id' => $customRequest->id,
                'topic' => $customRequest->topic,
                'status' => $customRequest->status,
                'createdAt' => $customRequest->created_at,
            ],
        ], 201);
    }

    public function index(Request $request): JsonResponse
    {
        $sessions = CustomSession::where('user_id', Auth::id())
            ->with('customRequest')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'sessions' => $sessions->items(),
            'pagination' => [
                'currentPage' => $sessions->currentPage(),
                'lastPage' => $sessions->lastPage(),
                'perPage' => $sessions->perPage(),
                'total' => $sessions->total(),
            ],
        ]);
    }

    public function show(CustomSession $session): JsonResponse
    {
        // Ensure user can only view their own sessions
        if ($session->user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        $session->load('customRequest');

        return response()->json([
            'session' => [
                'id' => $session->id,
                'title' => $session->title,
                'description' => $session->description,
                'transcript' => $session->transcript,
                'duration' => $session->duration,
                'formattedDuration' => $session->formatted_duration,
                'fileSize' => $session->file_size,
                'formattedFileSize' => $session->formatted_file_size,
                'voiceType' => $session->voice_type,
                'status' => $session->status,
                'generationMetadata' => $session->generation_metadata,
                'playCount' => $session->play_count,
                'generatedAt' => $session->generated_at,
                'failedAt' => $session->failed_at,
                'failureReason' => $session->failure_reason,
                'request' => [
                    'id' => $session->customRequest->id,
                    'topic' => $session->customRequest->topic,
                    'description' => $session->customRequest->description,
                    'sessionLength' => $session->customRequest->session_length,
                    'voicePreference' => $session->customRequest->voice_preference,
                    'specialNotes' => $session->customRequest->special_notes,
                    'status' => $session->customRequest->status,
                ],
            ],
        ]);
    }
}
