<?php

namespace App\Services\AI;

use App\Models\Receipt;
use App\Models\ReceiptGroup;
use App\Services\AI\VertexClient;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class ReceiptDeduplicationService
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Process a newly OCR'd receipt to check for duplicates
     * This runs immediately after OCR, before any matching
     */
    public function processReceipt(Receipt $newReceipt): ?ReceiptGroup
    {
        // Simple duplicate detection only - no grouping
        $duplicate = $this->findDuplicate($newReceipt);
        
        if ($duplicate) {
            \Log::info('Found duplicate receipt', [
                'new_receipt_id' => $newReceipt->id,
                'duplicate_receipt_id' => $duplicate->id,
                'merchant' => $newReceipt->merchant_name,
                'amount' => $newReceipt->total_amount
            ]);
            
            // Mark as duplicate in meta data
            $meta = $newReceipt->meta ?? [];
            $meta['is_duplicate'] = true;
            $meta['duplicate_of'] = $duplicate->id;
            $newReceipt->update(['meta' => $meta]);
            
            return null; // No grouping
        }

        return null;
    }

    /**
     * Find duplicate receipts using simple matching
     */
    private function findDuplicate(Receipt $newReceipt): ?Receipt
    {
        // Look for exact duplicates based on merchant, amount, and date
        $duplicates = Receipt::where('id', '!=', $newReceipt->id)
            ->where('user_id', $newReceipt->user_id)
            ->where('merchant_name', $newReceipt->merchant_name)
            ->where('total_amount', $newReceipt->total_amount)
            ->where('receipt_date', $newReceipt->receipt_date)
            ->whereDoesntHave('matches')
            ->get();

        return $duplicates->first();
    }


    /**
     * Process all unmatched receipts to find duplicates
     * This can be run as a cleanup job
     */
    public function processAllUnmatchedReceipts(): array
    {
        $results = [];
        $unmatchedReceipts = Receipt::whereDoesntHave('matches')
            ->orderBy('created_at', 'asc')
            ->get();

        foreach ($unmatchedReceipts as $receipt) {
            $duplicate = $this->findDuplicate($receipt);
            if ($duplicate) {
                // Mark as duplicate
                $meta = $receipt->meta ?? [];
                $meta['is_duplicate'] = true;
                $meta['duplicate_of'] = $duplicate->id;
                $receipt->update(['meta' => $meta]);
                
                $results[] = [
                    'receipt_id' => $receipt->id,
                    'duplicate_of' => $duplicate->id,
                    'merchant' => $receipt->merchant_name,
                    'amount' => $receipt->total_amount
                ];
            }
        }

        return $results;
    }

    /**
     * Check if a receipt is a duplicate of an existing one
     */
    public function isDuplicate(Receipt $receipt1, Receipt $receipt2): bool
    {
        // Same merchant, same amount, same date
        if ($receipt1->merchant_name === $receipt2->merchant_name &&
            abs($receipt1->total_amount - $receipt2->total_amount) < 0.01 &&
            $receipt1->receipt_date === $receipt2->receipt_date) {
            return true;
        }

        // Check line items for similarity
        $lines1 = $receipt1->lines->pluck('description')->sort()->values();
        $lines2 = $receipt2->lines->pluck('description')->sort()->values();
        
        if ($lines1->count() > 0 && $lines2->count() > 0) {
            $similarity = $lines1->intersect($lines2)->count() / max($lines1->count(), $lines2->count());
            if ($similarity > 0.8) {
                return true;
            }
        }

        return false;
    }

    /**
     * Detect split receipt patterns using AI-powered analysis
     * This works for any merchant, not just hardcoded patterns
     */
    private function detectKnownSplitPatterns(Receipt $newReceipt, Collection $existingReceipts): ?array
    {
        // First try hardcoded patterns for known cases
        $hardcodedResult = $this->detectHardcodedSplitPatterns($newReceipt, $existingReceipts);
        if ($hardcodedResult) {
            return $hardcodedResult;
        }

        // Then use AI to detect generic split patterns
        return $this->detectGenericSplitPatterns($newReceipt, $existingReceipts);
    }

    /**
     * Detect hardcoded split patterns (for known specific cases)
     */
    private function detectHardcodedSplitPatterns(Receipt $newReceipt, Collection $existingReceipts): ?array
    {
        // Known split receipt patterns
        $splitPatterns = [
            // Costco patterns - handle typos and variations
            'costco' => [
                'merchants' => [
                    'costco wholesale', 
                    'robert todd and son ltd', 
                    'robert todd & son ltd', 
                    'r todd and son',
                    'fobert todd and son ltd', // Handle typo
                    'robert todd and son',
                    'todd and son ltd'
                ],
                'description' => 'Costco split receipt pattern',
                'max_date_diff_days' => 365, // Allow up to 1 year difference
                'amount_tolerance' => 0.2 // Allow 20% difference in amounts
            ],
            // Add more hardcoded patterns as needed
        ];

        foreach ($splitPatterns as $patternName => $pattern) {
            $merchants = array_map('strtolower', $pattern['merchants']);
            $newMerchant = strtolower(trim($newReceipt->merchant_name ?? ''));
            
            if (!in_array($newMerchant, $merchants)) {
                continue;
            }

            // Look for receipts with complementary merchants and amounts
            foreach ($existingReceipts as $existingReceipt) {
                $existingMerchant = strtolower(trim($existingReceipt->merchant_name ?? ''));
                
                // Check if this is a complementary merchant in the same pattern
                if (!in_array($existingMerchant, $merchants) || $existingMerchant === $newMerchant) {
                    continue;
                }

                // Check date proximity (use pattern-specific tolerance)
                $maxDateDiff = $pattern['max_date_diff_days'] ?? 1;
                if ($newReceipt->receipt_date && $existingReceipt->receipt_date) {
                    $dateDiff = abs(Carbon::parse($newReceipt->receipt_date)->diffInDays(Carbon::parse($existingReceipt->receipt_date)));
                    if ($dateDiff > $maxDateDiff) {
                        continue;
                    }
                }

                // Check amount complementarity
                $totalAmount = $newReceipt->total_amount + $existingReceipt->total_amount;
                $amountComplementarity = $this->calculateAmountComplementarity($newReceipt->total_amount, $existingReceipt->total_amount);
                $amountTolerance = $pattern['amount_tolerance'] ?? 0.1; // Default 10% tolerance
                
                if ($amountComplementarity > (0.6 - $amountTolerance)) {
                    \Log::info('Detected hardcoded split pattern', [
                        'pattern' => $patternName,
                        'new_receipt' => $newReceipt->id,
                        'existing_receipt' => $existingReceipt->id,
                        'new_merchant' => $newReceipt->merchant_name,
                        'existing_merchant' => $existingReceipt->merchant_name,
                        'total_amount' => $totalAmount,
                        'complementarity' => $amountComplementarity
                    ]);

                    return [
                        'confidence' => 0.9, // High confidence for known patterns
                        'relationship_type' => 'split',
                        'related_receipt_ids' => [$existingReceipt->id],
                        'reasoning' => "Detected known split receipt pattern: {$pattern['description']}. Different merchant names ({$newReceipt->merchant_name} + {$existingReceipt->merchant_name}) but complementary amounts (£{$newReceipt->total_amount} + £{$existingReceipt->total_amount} = £{$totalAmount})",
                        'group_name' => "Split Receipt - " . ucfirst($patternName)
                    ];
                }
            }
        }

        return null;
    }

    /**
     * Detect generic split patterns using AI analysis
     * This works for any merchant, not just hardcoded ones
     */
    private function detectGenericSplitPatterns(Receipt $newReceipt, Collection $existingReceipts): ?array
    {
        // Look for receipts that might be split based on common indicators
        foreach ($existingReceipts as $existingReceipt) {
            // Skip if same merchant (not a split)
            if (strtolower(trim($newReceipt->merchant_name ?? '')) === strtolower(trim($existingReceipt->merchant_name ?? ''))) {
                continue;
            }

            // Check date proximity (within 3 days for generic detection)
            if ($newReceipt->receipt_date && $existingReceipt->receipt_date) {
                $dateDiff = abs(Carbon::parse($newReceipt->receipt_date)->diffInDays(Carbon::parse($existingReceipt->receipt_date)));
                if ($dateDiff > 3) {
                    continue;
                }
            }

            // Check amount complementarity
            $amountComplementarity = $this->calculateAmountComplementarity($newReceipt->total_amount, $existingReceipt->total_amount);
            
            // Look for strong indicators of split receipts
            $splitIndicators = $this->analyzeSplitIndicators($newReceipt, $existingReceipt);
            
            if ($amountComplementarity > 0.7 && $splitIndicators['score'] > 0.6) {
                \Log::info('Detected generic split pattern', [
                    'new_receipt' => $newReceipt->id,
                    'existing_receipt' => $existingReceipt->id,
                    'new_merchant' => $newReceipt->merchant_name,
                    'existing_merchant' => $existingReceipt->merchant_name,
                    'amount_complementarity' => $amountComplementarity,
                    'split_indicators' => $splitIndicators
                ]);

                return [
                    'confidence' => min(0.8, $amountComplementarity + $splitIndicators['score']),
                    'relationship_type' => 'split',
                    'related_receipt_ids' => [$existingReceipt->id],
                    'reasoning' => "Detected potential split receipt: Different merchant names ({$newReceipt->merchant_name} + {$existingReceipt->merchant_name}) with complementary amounts (£{$newReceipt->total_amount} + £{$existingReceipt->total_amount} = £" . ($newReceipt->total_amount + $existingReceipt->total_amount) . "). Indicators: {$splitIndicators['reasoning']}",
                    'group_name' => "Split Receipt - " . ($newReceipt->merchant_name ?: 'Unknown') . " + " . ($existingReceipt->merchant_name ?: 'Unknown')
                ];
            }
        }

        return null;
    }

    /**
     * Analyze indicators that suggest two receipts might be split
     */
    private function analyzeSplitIndicators(Receipt $receipt1, Receipt $receipt2): array
    {
        $indicators = [];
        $score = 0.0;

        // Check for similar receipt structure
        $lines1 = $receipt1->lines->count();
        $lines2 = $receipt2->lines->count();
        
        if ($lines1 > 0 && $lines2 > 0) {
            // If one has many lines and the other has few, might be itemized vs summary
            if (($lines1 > 5 && $lines2 <= 3) || ($lines2 > 5 && $lines1 <= 3)) {
                $indicators[] = 'One detailed, one summary receipt';
                $score += 0.3;
            }
        }

        // Check for complementary amounts (one much larger than the other)
        $ratio = $receipt1->total_amount / $receipt2->total_amount;
        if ($ratio > 2 || $ratio < 0.5) {
            $indicators[] = 'Complementary amounts (one much larger)';
            $score += 0.2;
        }

        // Check for round number totals when combined
        $total = $receipt1->total_amount + $receipt2->total_amount;
        if (abs($total - round($total)) < 0.01) {
            $indicators[] = 'Combined total is a round number';
            $score += 0.3;
        }

        // Check for similar line items (if any)
        if ($lines1 > 0 && $lines2 > 0) {
            $descriptions1 = $receipt1->lines->pluck('description')->map('strtolower');
            $descriptions2 = $receipt2->lines->pluck('description')->map('strtolower');
            $commonItems = $descriptions1->intersect($descriptions2)->count();
            $totalItems = $descriptions1->count() + $descriptions2->count();
            
            if ($commonItems > 0 && $commonItems / $totalItems > 0.3) {
                $indicators[] = 'Some common line items';
                $score += 0.2;
            }
        }

        return [
            'score' => min(1.0, $score),
            'indicators' => $indicators,
            'reasoning' => implode(', ', $indicators) ?: 'No specific indicators'
        ];
    }

    /**
     * Calculate how complementary two amounts are
     */
    private function calculateAmountComplementarity(float $amount1, float $amount2): float
    {
        $total = $amount1 + $amount2;
        
        // Check if amounts are complementary (sum to a round number)
        if (abs($total - round($total)) < 0.01) {
            return 0.9;
        }

        // Check if one is much smaller (might be a partial receipt)
        if ($amount1 < $amount2 * 0.3 || $amount2 < $amount1 * 0.3) {
            return 0.7;
        }

        // Check if amounts are similar (might be duplicates)
        $ratio = $amount1 / $amount2;
        if ($ratio > 0.8 && $ratio < 1.2) {
            return 0.6;
        }

        return 0.0;
    }
}
