<?php

namespace App\Services\AI;

use App\Models\Receipt;
use App\Models\ReceiptGroup;
use App\Services\AI\VertexClient;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class ReceiptDeduplicationService
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Process a newly OCR'd receipt to check for duplicates or split receipts
     * This runs immediately after OCR, before any matching
     */
    public function processReceipt(Receipt $newReceipt): ?ReceiptGroup
    {
        // Get all recent unmatched receipts (within last 30 days)
        $recentReceipts = Receipt::where('id', '!=', $newReceipt->id)
            ->whereNull('receipt_group_id')
            ->whereDoesntHave('matches')
            ->where(function($query) use ($newReceipt) {
                // Same user
                $query->where('user_id', $newReceipt->user_id);
                
                // Within 30 days
                if ($newReceipt->receipt_date) {
                    $query->whereBetween('receipt_date', [
                        Carbon::parse($newReceipt->receipt_date)->subDays(30),
                        Carbon::parse($newReceipt->receipt_date)->addDays(30)
                    ]);
                }
            })
            ->with(['lines'])
            ->get();

        if ($recentReceipts->isEmpty()) {
            return null;
        }

        // Use LLM to analyze if this receipt is related to any existing ones
        $analysis = $this->analyzeReceiptRelationships($newReceipt, $recentReceipts);

        if (!$analysis || $analysis['confidence'] < 0.7) {
            return null;
        }

        // Create or update group based on analysis
        return $this->createOrUpdateGroup($newReceipt, $recentReceipts, $analysis);
    }

    /**
     * Use LLM to analyze relationships between receipts
     */
    private function analyzeReceiptRelationships(Receipt $newReceipt, Collection $existingReceipts): ?array
    {
        // First, try rule-based detection for known patterns
        $ruleBasedResult = $this->detectKnownSplitPatterns($newReceipt, $existingReceipts);
        if ($ruleBasedResult) {
            return $ruleBasedResult;
        }

        $model = config('vertex.models.match', 'gemini-2.5-flash-lite');
        $system = 'You are an expert expense management assistant. Your task is to analyze a newly processed receipt and determine if it is related to any existing receipts.

RELATIONSHIP TYPES TO DETECT:
1. DUPLICATE RECEIPTS: Same receipt photographed multiple times
2. SPLIT RECEIPTS: One receipt photographed in multiple parts (CRITICAL: merchant names may be different!)
3. CARD + ITEMIZED: Card receipt + detailed itemized receipt from same transaction
4. RELATED PURCHASES: Multiple purchases at same location on same day

SPLIT RECEIPT INDICATORS (merchant names may differ):
- Different merchant names but same business (e.g., "COSTCO WHOLESALE" vs "ROBERT TODD AND SON LTD")
- Complementary amounts that sum to a logical total
- Same or very close dates
- Similar receipt structure or layout
- Sequential receipt numbers or transaction IDs
- Same location/address but different business name variations

ANALYSIS CRITERIA:
- MERCHANT SIMILARITY: Same business (handle name variations, different legal entities)
- AMOUNT RELATIONSHIPS: Exact match (duplicate), complementary (split), or related (same location)
- DATE PROXIMITY: Same or very close dates
- RECEIPT CONTENT: Similar line items, same transaction details, sequential numbers
- BUSINESS CONTEXT: Logical business relationships, same location
- RECEIPT STRUCTURE: Similar layout, formatting, or receipt style

OUTPUT JSON:
{
  "is_related": true/false,
  "confidence": 0.0-1.0,
  "relationship_type": "duplicate|split|card_itemized|related_purchase",
  "related_receipt_ids": [123, 124],
  "reasoning": "Detailed explanation",
  "group_name": "Suggested group name"
}

CONFIDENCE THRESHOLDS:
- 0.9-1.0: Very high confidence (exact duplicate or clear split)
- 0.7-0.9: High confidence (likely related, should group)
- 0.5-0.7: Medium confidence (possible relationship, review needed)
- 0.0-0.5: Low confidence (unlikely to be related)

IMPORTANT: Pay special attention to split receipts where merchant names are completely different but amounts are complementary and dates are close. These are often the same transaction split across different business entities or receipt parts.

RETURN is_related=true ONLY if confidence >= 0.7';

        $payload = [
            'new_receipt' => [
                'id' => $newReceipt->id,
                'date' => $newReceipt->receipt_date?->format('Y-m-d'),
                'merchant' => $newReceipt->merchant_name,
                'amount' => $newReceipt->total_amount,
                'currency' => $newReceipt->currency,
                'lines' => $newReceipt->lines->map(fn($line) => [
                    'description' => $line->description,
                    'quantity' => $line->quantity,
                    'unit_price' => $line->unit_price,
                    'line_total' => $line->line_total,
                ])->toArray(),
            ],
            'existing_receipts' => $existingReceipts->map(fn($receipt) => [
                'id' => $receipt->id,
                'date' => $receipt->receipt_date?->format('Y-m-d'),
                'merchant' => $receipt->merchant_name,
                'amount' => $receipt->total_amount,
                'currency' => $receipt->currency,
                'lines' => $receipt->lines->map(fn($line) => [
                    'description' => $line->description,
                    'quantity' => $line->quantity,
                    'unit_price' => $line->unit_price,
                    'line_total' => $line->line_total,
                ])->toArray(),
            ])->values()->all(),
        ];

        $prompt = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $resp = $this->vertex->generate($model, $prompt, $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($resp['json']) || !is_array($resp['json'])) {
            return null;
        }

        $result = $resp['json'];
        
        if (!isset($result['is_related']) || !$result['is_related']) {
            return null;
        }

        return [
            'confidence' => (float) ($result['confidence'] ?? 0),
            'relationship_type' => $result['relationship_type'] ?? 'unknown',
            'related_receipt_ids' => $result['related_receipt_ids'] ?? [],
            'reasoning' => $result['reasoning'] ?? 'AI detected relationship',
            'group_name' => $result['group_name'] ?? 'Grouped Receipts',
        ];
    }

    /**
     * Create or update a receipt group based on analysis
     */
    private function createOrUpdateGroup(Receipt $newReceipt, Collection $existingReceipts, array $analysis): ReceiptGroup
    {
        $relatedReceipts = $existingReceipts->whereIn('id', $analysis['related_receipt_ids']);
        
        // Check if any of the related receipts are already in a group
        $existingGroup = null;
        foreach ($relatedReceipts as $receipt) {
            if ($receipt->receipt_group_id) {
                $existingGroup = ReceiptGroup::find($receipt->receipt_group_id);
                break;
            }
        }

        if ($existingGroup) {
            // Add new receipt to existing group
            $newReceipt->update(['receipt_group_id' => $existingGroup->id]);
            
            \Log::info('Added receipt to existing group', [
                'receipt_id' => $newReceipt->id,
                'group_id' => $existingGroup->id,
                'relationship_type' => $analysis['relationship_type']
            ]);
            
            return $existingGroup;
        } else {
            // Create new group with all related receipts
            $group = ReceiptGroup::create([
                'name' => $analysis['group_name'],
                'description' => $analysis['reasoning'],
                'grouping_method' => 'ai_auto',
                'confidence_score' => $analysis['confidence'],
                'grouping_reason' => $analysis['reasoning'],
                'user_id' => $newReceipt->user_id,
            ]);

            // Add all related receipts to the group
            $allReceiptIds = array_merge([$newReceipt->id], $analysis['related_receipt_ids']);
            Receipt::whereIn('id', $allReceiptIds)
                ->update(['receipt_group_id' => $group->id]);

            \Log::info('Created new receipt group', [
                'group_id' => $group->id,
                'receipt_count' => count($allReceiptIds),
                'relationship_type' => $analysis['relationship_type'],
                'confidence' => $analysis['confidence']
            ]);

            return $group;
        }
    }

    /**
     * Process all unmatched receipts to find relationships
     * This can be run as a cleanup job
     */
    public function processAllUnmatchedReceipts(): array
    {
        $results = [];
        $unmatchedReceipts = Receipt::whereNull('receipt_group_id')
            ->whereDoesntHave('matches')
            ->with(['lines'])
            ->orderBy('created_at', 'asc')
            ->get();

        foreach ($unmatchedReceipts as $receipt) {
            $group = $this->processReceipt($receipt);
            if ($group) {
                $results[] = [
                    'receipt_id' => $receipt->id,
                    'group_id' => $group->id,
                    'group_name' => $group->name,
                ];
            }
        }

        return $results;
    }

    /**
     * Process a batch of receipts for deduplication
     * This is more efficient than processing individual receipts
     */
    public function processBatch(array $receiptIds): array
    {
        $results = [];
        
        // Get all receipts in the batch
        $receipts = Receipt::whereIn('id', $receiptIds)
            ->whereNull('receipt_group_id')
            ->whereDoesntHave('matches')
            ->with(['lines'])
            ->get();

        if ($receipts->isEmpty()) {
            return $results;
        }

        \Log::info('Processing batch for deduplication', [
            'receipt_count' => $receipts->count(),
            'receipt_ids' => $receiptIds
        ]);

        // Group receipts by user and date range for more efficient processing
        $groupedByUser = $receipts->groupBy('user_id');
        
        foreach ($groupedByUser as $userId => $userReceipts) {
            $results = array_merge($results, $this->processUserReceipts($userReceipts));
        }

        \Log::info('Batch deduplication completed', [
            'groups_created' => count($results),
            'receipt_count' => $receipts->count()
        ]);

        return $results;
    }

    /**
     * Process receipts for a specific user
     */
    private function processUserReceipts(Collection $receipts): array
    {
        $results = [];
        
        // Sort receipts by date for better grouping
        $receipts = $receipts->sortBy('receipt_date');
        
        foreach ($receipts as $receipt) {
            // Skip if already grouped
            if ($receipt->receipt_group_id) {
                continue;
            }

            // Get recent receipts for this user (within 30 days)
            $recentReceipts = Receipt::where('id', '!=', $receipt->id)
                ->where('user_id', $receipt->user_id)
                ->whereNull('receipt_group_id')
                ->whereDoesntHave('matches')
                ->where(function($query) use ($receipt) {
                    if ($receipt->receipt_date) {
                        $query->whereBetween('receipt_date', [
                            Carbon::parse($receipt->receipt_date)->subDays(30),
                            Carbon::parse($receipt->receipt_date)->addDays(30)
                        ]);
                    }
                })
                ->with(['lines'])
                ->get();

            if ($recentReceipts->isEmpty()) {
                continue;
            }

            // Use LLM to analyze relationships
            $analysis = $this->analyzeReceiptRelationships($receipt, $recentReceipts);

            if (!$analysis || $analysis['confidence'] < 0.7) {
                continue;
            }

            // Create or update group
            $group = $this->createOrUpdateGroup($receipt, $recentReceipts, $analysis);
            
            if ($group) {
                $results[] = [
                    'receipt_id' => $receipt->id,
                    'group_id' => $group->id,
                    'group_name' => $group->name,
                    'confidence' => $analysis['confidence'],
                    'relationship_type' => $analysis['relationship_type']
                ];
            }
        }

        return $results;
    }

    /**
     * Check if a receipt is a duplicate of an existing one
     */
    public function isDuplicate(Receipt $receipt1, Receipt $receipt2): bool
    {
        // Same merchant, same amount, same date
        if ($receipt1->merchant_name === $receipt2->merchant_name &&
            abs($receipt1->total_amount - $receipt2->total_amount) < 0.01 &&
            $receipt1->receipt_date === $receipt2->receipt_date) {
            return true;
        }

        // Check line items for similarity
        $lines1 = $receipt1->lines->pluck('description')->sort()->values();
        $lines2 = $receipt2->lines->pluck('description')->sort()->values();
        
        if ($lines1->count() > 0 && $lines2->count() > 0) {
            $similarity = $lines1->intersect($lines2)->count() / max($lines1->count(), $lines2->count());
            if ($similarity > 0.8) {
                return true;
            }
        }

        return false;
    }

    /**
     * Detect split receipt patterns using AI-powered analysis
     * This works for any merchant, not just hardcoded patterns
     */
    private function detectKnownSplitPatterns(Receipt $newReceipt, Collection $existingReceipts): ?array
    {
        // First try hardcoded patterns for known cases
        $hardcodedResult = $this->detectHardcodedSplitPatterns($newReceipt, $existingReceipts);
        if ($hardcodedResult) {
            return $hardcodedResult;
        }

        // Then use AI to detect generic split patterns
        return $this->detectGenericSplitPatterns($newReceipt, $existingReceipts);
    }

    /**
     * Detect hardcoded split patterns (for known specific cases)
     */
    private function detectHardcodedSplitPatterns(Receipt $newReceipt, Collection $existingReceipts): ?array
    {
        // Known split receipt patterns
        $splitPatterns = [
            // Costco patterns - handle typos and variations
            'costco' => [
                'merchants' => [
                    'costco wholesale', 
                    'robert todd and son ltd', 
                    'robert todd & son ltd', 
                    'r todd and son',
                    'fobert todd and son ltd', // Handle typo
                    'robert todd and son',
                    'todd and son ltd'
                ],
                'description' => 'Costco split receipt pattern',
                'max_date_diff_days' => 365, // Allow up to 1 year difference
                'amount_tolerance' => 0.2 // Allow 20% difference in amounts
            ],
            // Add more hardcoded patterns as needed
        ];

        foreach ($splitPatterns as $patternName => $pattern) {
            $merchants = array_map('strtolower', $pattern['merchants']);
            $newMerchant = strtolower(trim($newReceipt->merchant_name ?? ''));
            
            if (!in_array($newMerchant, $merchants)) {
                continue;
            }

            // Look for receipts with complementary merchants and amounts
            foreach ($existingReceipts as $existingReceipt) {
                $existingMerchant = strtolower(trim($existingReceipt->merchant_name ?? ''));
                
                // Check if this is a complementary merchant in the same pattern
                if (!in_array($existingMerchant, $merchants) || $existingMerchant === $newMerchant) {
                    continue;
                }

                // Check date proximity (use pattern-specific tolerance)
                $maxDateDiff = $pattern['max_date_diff_days'] ?? 1;
                if ($newReceipt->receipt_date && $existingReceipt->receipt_date) {
                    $dateDiff = abs(Carbon::parse($newReceipt->receipt_date)->diffInDays(Carbon::parse($existingReceipt->receipt_date)));
                    if ($dateDiff > $maxDateDiff) {
                        continue;
                    }
                }

                // Check amount complementarity
                $totalAmount = $newReceipt->total_amount + $existingReceipt->total_amount;
                $amountComplementarity = $this->calculateAmountComplementarity($newReceipt->total_amount, $existingReceipt->total_amount);
                $amountTolerance = $pattern['amount_tolerance'] ?? 0.1; // Default 10% tolerance
                
                if ($amountComplementarity > (0.6 - $amountTolerance)) {
                    \Log::info('Detected hardcoded split pattern', [
                        'pattern' => $patternName,
                        'new_receipt' => $newReceipt->id,
                        'existing_receipt' => $existingReceipt->id,
                        'new_merchant' => $newReceipt->merchant_name,
                        'existing_merchant' => $existingReceipt->merchant_name,
                        'total_amount' => $totalAmount,
                        'complementarity' => $amountComplementarity
                    ]);

                    return [
                        'confidence' => 0.9, // High confidence for known patterns
                        'relationship_type' => 'split',
                        'related_receipt_ids' => [$existingReceipt->id],
                        'reasoning' => "Detected known split receipt pattern: {$pattern['description']}. Different merchant names ({$newReceipt->merchant_name} + {$existingReceipt->merchant_name}) but complementary amounts (£{$newReceipt->total_amount} + £{$existingReceipt->total_amount} = £{$totalAmount})",
                        'group_name' => "Split Receipt - " . ucfirst($patternName)
                    ];
                }
            }
        }

        return null;
    }

    /**
     * Detect generic split patterns using AI analysis
     * This works for any merchant, not just hardcoded ones
     */
    private function detectGenericSplitPatterns(Receipt $newReceipt, Collection $existingReceipts): ?array
    {
        // Look for receipts that might be split based on common indicators
        foreach ($existingReceipts as $existingReceipt) {
            // Skip if same merchant (not a split)
            if (strtolower(trim($newReceipt->merchant_name ?? '')) === strtolower(trim($existingReceipt->merchant_name ?? ''))) {
                continue;
            }

            // Check date proximity (within 3 days for generic detection)
            if ($newReceipt->receipt_date && $existingReceipt->receipt_date) {
                $dateDiff = abs(Carbon::parse($newReceipt->receipt_date)->diffInDays(Carbon::parse($existingReceipt->receipt_date)));
                if ($dateDiff > 3) {
                    continue;
                }
            }

            // Check amount complementarity
            $amountComplementarity = $this->calculateAmountComplementarity($newReceipt->total_amount, $existingReceipt->total_amount);
            
            // Look for strong indicators of split receipts
            $splitIndicators = $this->analyzeSplitIndicators($newReceipt, $existingReceipt);
            
            if ($amountComplementarity > 0.7 && $splitIndicators['score'] > 0.6) {
                \Log::info('Detected generic split pattern', [
                    'new_receipt' => $newReceipt->id,
                    'existing_receipt' => $existingReceipt->id,
                    'new_merchant' => $newReceipt->merchant_name,
                    'existing_merchant' => $existingReceipt->merchant_name,
                    'amount_complementarity' => $amountComplementarity,
                    'split_indicators' => $splitIndicators
                ]);

                return [
                    'confidence' => min(0.8, $amountComplementarity + $splitIndicators['score']),
                    'relationship_type' => 'split',
                    'related_receipt_ids' => [$existingReceipt->id],
                    'reasoning' => "Detected potential split receipt: Different merchant names ({$newReceipt->merchant_name} + {$existingReceipt->merchant_name}) with complementary amounts (£{$newReceipt->total_amount} + £{$existingReceipt->total_amount} = £" . ($newReceipt->total_amount + $existingReceipt->total_amount) . "). Indicators: {$splitIndicators['reasoning']}",
                    'group_name' => "Split Receipt - " . ($newReceipt->merchant_name ?: 'Unknown') . " + " . ($existingReceipt->merchant_name ?: 'Unknown')
                ];
            }
        }

        return null;
    }

    /**
     * Analyze indicators that suggest two receipts might be split
     */
    private function analyzeSplitIndicators(Receipt $receipt1, Receipt $receipt2): array
    {
        $indicators = [];
        $score = 0.0;

        // Check for similar receipt structure
        $lines1 = $receipt1->lines->count();
        $lines2 = $receipt2->lines->count();
        
        if ($lines1 > 0 && $lines2 > 0) {
            // If one has many lines and the other has few, might be itemized vs summary
            if (($lines1 > 5 && $lines2 <= 3) || ($lines2 > 5 && $lines1 <= 3)) {
                $indicators[] = 'One detailed, one summary receipt';
                $score += 0.3;
            }
        }

        // Check for complementary amounts (one much larger than the other)
        $ratio = $receipt1->total_amount / $receipt2->total_amount;
        if ($ratio > 2 || $ratio < 0.5) {
            $indicators[] = 'Complementary amounts (one much larger)';
            $score += 0.2;
        }

        // Check for round number totals when combined
        $total = $receipt1->total_amount + $receipt2->total_amount;
        if (abs($total - round($total)) < 0.01) {
            $indicators[] = 'Combined total is a round number';
            $score += 0.3;
        }

        // Check for similar line items (if any)
        if ($lines1 > 0 && $lines2 > 0) {
            $descriptions1 = $receipt1->lines->pluck('description')->map('strtolower');
            $descriptions2 = $receipt2->lines->pluck('description')->map('strtolower');
            $commonItems = $descriptions1->intersect($descriptions2)->count();
            $totalItems = $descriptions1->count() + $descriptions2->count();
            
            if ($commonItems > 0 && $commonItems / $totalItems > 0.3) {
                $indicators[] = 'Some common line items';
                $score += 0.2;
            }
        }

        return [
            'score' => min(1.0, $score),
            'indicators' => $indicators,
            'reasoning' => implode(', ', $indicators) ?: 'No specific indicators'
        ];
    }

    /**
     * Calculate how complementary two amounts are
     */
    private function calculateAmountComplementarity(float $amount1, float $amount2): float
    {
        $total = $amount1 + $amount2;
        
        // Check if amounts are complementary (sum to a round number)
        if (abs($total - round($total)) < 0.01) {
            return 0.9;
        }

        // Check if one is much smaller (might be a partial receipt)
        if ($amount1 < $amount2 * 0.3 || $amount2 < $amount1 * 0.3) {
            return 0.7;
        }

        // Check if amounts are similar (might be duplicates)
        $ratio = $amount1 / $amount2;
        if ($ratio > 0.8 && $ratio < 1.2) {
            return 0.6;
        }

        return 0.0;
    }
}
