<?php

namespace App\Services\AI;

use App\Models\Receipt;
use App\Models\ReceiptGroup;
use App\Services\AI\VertexClient;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class ReceiptDeduplicationService
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Process a newly OCR'd receipt to check for duplicates
     * This runs immediately after OCR, before any matching
     */
    public function processReceipt(Receipt $newReceipt): ?ReceiptGroup
    {
        // Simple duplicate detection only - no grouping
        $duplicate = $this->findDuplicate($newReceipt);
        
        if ($duplicate) {
            \Log::info('Found duplicate receipt', [
                'new_receipt_id' => $newReceipt->id,
                'duplicate_receipt_id' => $duplicate->id,
                'merchant' => $newReceipt->merchant_name,
                'amount' => $newReceipt->total_amount
            ]);
            
            // Mark as duplicate in meta data
            $meta = $newReceipt->meta ?? [];
            $meta['is_duplicate'] = true;
            $meta['duplicate_of'] = $duplicate->id;
            $newReceipt->update(['meta' => $meta]);
            
            return null; // No grouping
        }

        return null;
    }

    /**
     * Find duplicate receipts using simple matching
     */
    private function findDuplicate(Receipt $newReceipt): ?Receipt
    {
        // Look for exact duplicates based on merchant, amount, and date
        $duplicates = Receipt::where('id', '!=', $newReceipt->id)
            ->where('user_id', $newReceipt->user_id)
            ->where('merchant_name', $newReceipt->merchant_name)
            ->where('total_amount', $newReceipt->total_amount)
            ->where('receipt_date', $newReceipt->receipt_date)
            ->whereDoesntHave('matches')
            ->get();

        return $duplicates->first();
    }


    /**
     * Process all unmatched receipts to find duplicates
     * This can be run as a cleanup job
     */
    public function processAllUnmatchedReceipts(): array
    {
        $results = [];
        $unmatchedReceipts = Receipt::whereDoesntHave('matches')
            ->orderBy('created_at', 'asc')
            ->get();

        foreach ($unmatchedReceipts as $receipt) {
            $duplicate = $this->findDuplicate($receipt);
            if ($duplicate) {
                // Mark as duplicate
                $meta = $receipt->meta ?? [];
                $meta['is_duplicate'] = true;
                $meta['duplicate_of'] = $duplicate->id;
                $receipt->update(['meta' => $meta]);
                
                $results[] = [
                    'receipt_id' => $receipt->id,
                    'duplicate_of' => $duplicate->id,
                    'merchant' => $receipt->merchant_name,
                    'amount' => $receipt->total_amount
                ];
            }
        }

        return $results;
    }

    /**
     * Check if a receipt is a duplicate of an existing one
     */
    public function isDuplicate(Receipt $receipt1, Receipt $receipt2): bool
    {
        // Same merchant, same amount, same date
        if ($receipt1->merchant_name === $receipt2->merchant_name &&
            abs($receipt1->total_amount - $receipt2->total_amount) < 0.01 &&
            $receipt1->receipt_date === $receipt2->receipt_date) {
            return true;
        }

        // Check line items for similarity
        $lines1 = $receipt1->lines->pluck('description')->sort()->values();
        $lines2 = $receipt2->lines->pluck('description')->sort()->values();
        
        if ($lines1->count() > 0 && $lines2->count() > 0) {
            $similarity = $lines1->intersect($lines2)->count() / max($lines1->count(), $lines2->count());
            if ($similarity > 0.8) {
                return true;
            }
        }

        return false;
    }
}
