# Stripe Integration Setup Guide

This guide will walk you through setting up Stripe for your Vybe Photo subscription platform.

## Overview

Vybe Photo uses Stripe for subscription billing with three paid tiers:
- **Starter**: £10/month - 1,000 photos, 5GB storage, subdomain only
- **Professional**: £25/month - 10,000 photos, 50GB storage, custom domain
- **Business**: £75/month - 30,000 photos, 150GB storage, multiple domains

All new subscriptions include a **14-day free trial** before charging begins.

## Prerequisites

1. A Stripe account (sign up at https://stripe.com)
2. Your API keys from the Stripe Dashboard

---

## Step 1: Get Your Stripe API Keys

You've already provided your publishable key:
```
pk_test_51SQqD1FSPBHdTZaUQjQXLLiGS0HVDzNHMb4IV8ZqEeJ8vYZEaAqHzhLZp7PLPClfvl8vO4YyY0SowRJSR8cImhOJ00lPJyIDJR
```

You also need your **Secret Key**:

1. Go to https://dashboard.stripe.com/test/apikeys
2. Find your "Secret key" (starts with `sk_test_`)
3. Click "Reveal test key" to copy it

**Security Note**: Never commit your secret key to version control. Store it in environment variables only.

---

## Step 2: Create Products and Prices in Stripe

### Option A: Using Stripe Dashboard (Recommended)

1. Go to https://dashboard.stripe.com/test/products
2. Click **"+ Add product"**

#### Create Starter Plan:
- **Name**: Vybe Photo - Starter
- **Description**: 1,000 photos, 5GB storage, unlimited albums, subdomain only
- **Pricing model**: Recurring
- **Price**: £10.00 GBP
- **Billing period**: Monthly
- Click **"Save product"**
- Copy the **Price ID** (starts with `price_`) - you'll need this

#### Create Professional Plan:
- **Name**: Vybe Photo - Professional
- **Description**: 10,000 photos, 50GB storage, unlimited albums, custom domain support
- **Pricing model**: Recurring
- **Price**: £25.00 GBP
- **Billing period**: Monthly
- Click **"Save product"**
- Copy the **Price ID**

#### Create Business Plan:
- **Name**: Vybe Photo - Business
- **Description**: 30,000 photos, 150GB storage, unlimited albums, multiple custom domains
- **Pricing model**: Recurring
- **Price**: £75.00 GBP
- **Billing period**: Monthly
- Click **"Save product"**
- Copy the **Price ID**

### Option B: Using Stripe CLI

Install the Stripe CLI and run:

```bash
# Create Starter product and price
stripe products create --name="Vybe Photo - Starter" --description="1,000 photos, 5GB storage"
stripe prices create --product=prod_XXXXX --unit-amount=1000 --currency=gbp --recurring[interval]=month

# Create Professional product and price
stripe products create --name="Vybe Photo - Professional" --description="10,000 photos, 50GB storage, custom domain"
stripe prices create --product=prod_YYYYY --unit-amount=2500 --currency=gbp --recurring[interval]=month

# Create Business product and price
stripe products create --name="Vybe Photo - Business" --description="30,000 photos, 150GB storage, multiple domains"
stripe prices create --product=prod_ZZZZZ --unit-amount=7500 --currency=gbp --recurring[interval]=month
```

---

## Step 3: Configure Environment Variables

Create or update your `.env` file with the following:

```bash
# Stripe Configuration
STRIPE_PUBLISHABLE_KEY=pk_test_51SQqD1FSPBHdTZaUQjQXLLiGS0HVDzNHMb4IV8ZqEeJ8vYZEaAqHzhLZp7PLPClfvl8vO4YyY0SowRJSR8cImhOJ00lPJyIDJR
STRIPE_SECRET_KEY=sk_test_YOUR_SECRET_KEY_HERE

# Stripe Price IDs (from Step 2)
STRIPE_PRICE_STARTER=price_1234567890abcdef
STRIPE_PRICE_PROFESSIONAL=price_abcdef1234567890
STRIPE_PRICE_BUSINESS=price_fedcba0987654321

# Stripe Webhook Secret (we'll get this in Step 4)
STRIPE_WEBHOOK_SECRET=whsec_YOUR_WEBHOOK_SECRET

# Application URL (used for redirects)
BASE_URL=http://localhost:3000
```

---

## Step 4: Set Up Webhook Endpoint

Stripe webhooks notify your application about subscription events (payments, cancellations, etc.).

### For Local Development:

1. Install Stripe CLI: https://stripe.com/docs/stripe-cli
2. Run the webhook listener:
   ```bash
   stripe listen --forward-to localhost:3000/webhooks/stripe
   ```
3. Copy the webhook signing secret (starts with `whsec_`) to your `.env` file

### For Production:

1. Go to https://dashboard.stripe.com/test/webhooks
2. Click **"+ Add endpoint"**
3. Enter your webhook URL: `https://yourdomain.com/webhooks/stripe`
4. Select events to listen to:
   - `customer.subscription.created`
   - `customer.subscription.updated`
   - `customer.subscription.deleted`
   - `customer.subscription.trial_will_end`
   - `invoice.paid`
   - `invoice.payment_failed`
   - `customer.created`
   - `customer.updated`
5. Click **"Add endpoint"**
6. Click on your new endpoint and reveal the "Signing secret"
7. Add it to your environment variables as `STRIPE_WEBHOOK_SECRET`

---

## Step 5: Enable Stripe Billing Portal

The billing portal allows customers to update payment methods, view invoices, and manage subscriptions.

1. Go to https://dashboard.stripe.com/test/settings/billing/portal
2. Click **"Activate test link"**
3. Configure portal settings:
   - **Allow customers to**: Update payment methods, View invoices
   - **Subscription cancellation**: Allow customers to cancel (optional)
   - **Proration**: Enable proration for upgrades/downgrades
4. Save settings

---

## Step 6: Test the Integration

### Start Your Server:

```bash
npm start
```

### Test Flow:

1. **Create a user account**:
   - Register at http://localhost:3000/auth/register
   - Or on your subdomain: http://yourusername.localhost:3000/auth/register

2. **Access subscription page**:
   - Navigate to `/admin/subscription`
   - You should see your current plan and usage

3. **Test plan upgrade**:
   - Click "Upgrade to Professional"
   - You'll be redirected to Stripe Checkout
   - Use Stripe test card: `4242 4242 4242 4242`
   - Expiry: Any future date
   - CVC: Any 3 digits
   - ZIP: Any 5 digits

4. **Test webhook events**:
   - After completing checkout, check your server logs
   - You should see `[STRIPE WEBHOOK]` messages
   - Check that your subscription tier updated in the database

5. **Test billing portal**:
   - Click "💳 Payment Info" on subscription page
   - You should be redirected to Stripe's billing portal
   - Test updating payment method (use another test card)

### Test Cards:

- **Success**: `4242 4242 4242 4242`
- **Declined**: `4000 0000 0000 0002`
- **Requires authentication**: `4000 0025 0000 3155`
- More test cards: https://stripe.com/docs/testing

---

## Step 7: Production Deployment

### Switch to Live Mode:

1. Get your live API keys from https://dashboard.stripe.com/apikeys
2. Update environment variables with live keys:
   ```bash
   STRIPE_PUBLISHABLE_KEY=pk_live_YOUR_LIVE_KEY
   STRIPE_SECRET_KEY=sk_live_YOUR_LIVE_KEY
   ```

3. Create live products and prices (repeat Step 2 in live mode)
4. Update price IDs in environment variables

5. Set up live webhook endpoint:
   - Go to https://dashboard.stripe.com/webhooks
   - Add endpoint: `https://yourdomain.com/webhooks/stripe`
   - Update `STRIPE_WEBHOOK_SECRET` with live webhook secret

6. Activate live billing portal:
   - Go to https://dashboard.stripe.com/settings/billing/portal
   - Configure and activate

### Security Checklist:

- ✅ Webhook signatures are verified
- ✅ Secret keys are in environment variables (never in code)
- ✅ HTTPS is enabled for production
- ✅ CSP headers include Stripe domains
- ✅ Rate limiting is enabled on payment endpoints

---

## Subscription Flow

### New User Sign-Up:
1. User registers an account (default: Starter tier with 14-day trial)
2. No payment required during trial period
3. 3 days before trial ends, Stripe sends `subscription.trial_will_end` event
4. On trial end, Stripe attempts to charge the card
5. If successful: `invoice.paid` → subscription becomes `active`
6. If failed: `invoice.payment_failed` → subscription becomes `past_due`

### Plan Changes:
1. User clicks upgrade/downgrade button
2. If first payment: Redirected to Stripe Checkout
3. If existing subscription: Immediate tier change with proration
4. Webhook updates user's tier and limits in database

### Cancellations:
1. User clicks "Cancel Subscription"
2. Subscription is set to cancel at period end
3. User retains access until end of billing period
4. On period end: Stripe sends `subscription.deleted` event
5. User is downgraded to free tier

---

## Troubleshooting

### Webhooks Not Working:

1. Check webhook secret matches your environment variable
2. Verify webhook URL is accessible (use `stripe listen` for local dev)
3. Check server logs for error messages
4. Test webhook delivery in Stripe Dashboard

### Payments Failing:

1. Verify price IDs match your Stripe products
2. Check currency is set to GBP
3. Ensure customer has payment method attached
4. Review Stripe logs for specific error messages

### Database Not Updating:

1. Check webhook handler is processing events
2. Verify `billing_events` table is being populated
3. Check for database connection errors
4. Review `usage_logs` table for audit trail

---

## Monitoring

### Key Metrics to Track:

- **MRR (Monthly Recurring Revenue)**: Track in Stripe Dashboard
- **Churn Rate**: Monitor cancelled subscriptions
- **Trial Conversions**: Track trial → paid conversions
- **Payment Failures**: Set up alerts for failed payments

### Stripe Dashboard Sections:

- **Payments**: View all transactions
- **Subscriptions**: Monitor active subscriptions
- **Customers**: Manage customer accounts
- **Analytics**: Revenue reports and metrics
- **Logs**: Debug webhook and API issues

---

## Support

### Stripe Resources:
- Documentation: https://stripe.com/docs
- API Reference: https://stripe.com/docs/api
- Support: https://support.stripe.com

### Common Issues:
- **Invalid API Key**: Check environment variables are loaded
- **Webhook Signature Mismatch**: Verify webhook secret is correct
- **Payment Declined**: User's card was declined (they need to update)
- **Subscription Not Found**: Check Stripe Customer ID is stored correctly

---

## Next Steps

1. ✅ Set up Stripe products and prices
2. ✅ Configure environment variables
3. ✅ Test subscription flow end-to-end
4. ✅ Set up monitoring and alerts
5. ⬜ Add email notifications for billing events
6. ⬜ Implement usage-based billing (optional)
7. ⬜ Add promotional codes/coupons (optional)
8. ⬜ Set up tax collection (if required)

---

## Additional Features

### Email Notifications:

Add email notifications for:
- Trial ending soon
- Payment successful
- Payment failed
- Subscription cancelled

Implement in `src/stripe-webhook.js` by integrating with your email service (SendGrid, Mailgun, etc.)

### Promotional Codes:

Create promo codes in Stripe Dashboard:
1. Go to https://dashboard.stripe.com/coupons
2. Create coupon (e.g., 20% off first 3 months)
3. Users can apply at checkout

### Usage-Based Billing:

For advanced setups, you can charge based on actual usage:
- Track photo uploads/storage in real-time
- Report usage to Stripe using metered billing
- Charge overages beyond plan limits

---

## Summary

Your Stripe integration includes:
- ✅ Subscription management (create, update, cancel)
- ✅ 14-day free trials
- ✅ Automatic billing and payment retries
- ✅ Self-service billing portal
- ✅ Webhook event handling
- ✅ Usage limit enforcement
- ✅ Secure payment processing

You're now ready to accept payments! 🎉

For questions or issues, refer to the Stripe documentation or check the application logs for detailed error messages.


