<x-app-layout>
    <x-slot name="header">
        <h2 class="font-semibold text-xl text-gray-800 leading-tight">
            {{ __('Library') }}
        </h2>
    </x-slot>

    <div class="py-12">
        <div class="max-w-7xl mx-auto sm:px-6 lg:px-8">
            <div class="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                <div class="p-6 text-gray-900">
                    <!-- Filters -->
                    <div class="mb-6">
                        <div class="flex flex-col sm:flex-row gap-4">
                            <div class="flex-1">
                                <label for="search" class="block text-sm font-medium text-gray-700">Search</label>
                                <input type="text" id="search" name="search" placeholder="Search tracks..." 
                                       class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            </div>
                            <div class="w-full sm:w-48">
                                <label for="category" class="block text-sm font-medium text-gray-700">Category</label>
                                <select id="category" name="category" 
                                        class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                    <option value="">All Categories</option>
                                </select>
                            </div>
                            <div class="w-full sm:w-48">
                                <label for="duration" class="block text-sm font-medium text-gray-700">Duration</label>
                                <select id="duration" name="duration" 
                                        class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                    <option value="">Any Duration</option>
                                    <option value="0-10">0-10 minutes</option>
                                    <option value="10-20">10-20 minutes</option>
                                    <option value="20+">20+ minutes</option>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Tracks Grid -->
                    <div id="tracks-container" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        <!-- Loading state -->
                        <div class="col-span-full text-center py-12">
                            <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600 mx-auto"></div>
                            <p class="mt-4 text-gray-500">Loading tracks...</p>
                        </div>
                    </div>

                    <!-- Empty State (hidden by default) -->
                    <div id="empty-state" class="hidden text-center py-12">
                        <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3" />
                        </svg>
                        <h3 class="mt-2 text-sm font-medium text-gray-900">No tracks found</h3>
                        <p class="mt-1 text-sm text-gray-500">Try adjusting your search or filter criteria.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            let currentPage = 1;
            let isLoading = false;
            let hasMorePages = true;

            // Load initial data
            loadCategories();
            loadTracks();

            // Set up event listeners for filters
            document.getElementById('search').addEventListener('input', debounce(handleSearch, 300));
            document.getElementById('category').addEventListener('change', handleFilterChange);
            document.getElementById('duration').addEventListener('change', handleFilterChange);

            function loadCategories() {
                fetch('/api/v1/library/categories')
                    .then(response => response.json())
                    .then(data => {
                        if (data.data) {
                            populateCategoryFilter(data.data);
                        }
                    })
                    .catch(error => {
                        console.error('Error loading categories:', error);
                    });
            }

            function populateCategoryFilter(categories) {
                const select = document.getElementById('category');
                // Keep the "All Categories" option
                select.innerHTML = '<option value="">All Categories</option>';
                
                categories.forEach(category => {
                    const option = document.createElement('option');
                    option.value = category.id;
                    option.textContent = category.name;
                    select.appendChild(option);
                });
            }

            function loadTracks(page = 1, append = false) {
                if (isLoading) return;
                
                isLoading = true;
                showLoadingState(append);

                const params = new URLSearchParams({
                    page: page,
                    per_page: 15
                });

                // Add search query
                const searchQuery = document.getElementById('search').value.trim();
                if (searchQuery) {
                    params.append('q', searchQuery);
                }

                // Add category filter
                const categoryId = document.getElementById('category').value;
                if (categoryId) {
                    params.append('category_id', categoryId);
                }

                // Add duration filter
                const duration = document.getElementById('duration').value;
                if (duration) {
                    params.append('duration', duration);
                }

                fetch(`/api/v1/library/tracks?${params.toString()}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.data && data.data.length > 0) {
                            displayTracks(data.data, append);
                            currentPage = page;
                            hasMorePages = data.next_page_url !== null;
                        } else {
                            showEmptyState();
                        }
                    })
                    .catch(error => {
                        console.error('Error loading tracks:', error);
                        showError('Failed to load tracks');
                    })
                    .finally(() => {
                        isLoading = false;
                        hideLoadingState();
                    });
            }

            function displayTracks(tracks, append = false) {
                const container = document.getElementById('tracks-container');
                
                if (!append) {
                    container.innerHTML = '';
                }

                tracks.forEach(track => {
                    const trackElement = createTrackElement(track);
                    container.appendChild(trackElement);
                });

                // Show/hide empty state
                if (container.children.length === 0) {
                    showEmptyState();
                } else {
                    hideEmptyState();
                }
            }

            function createTrackElement(track) {
                const div = document.createElement('div');
                div.className = 'bg-white rounded-lg shadow-md overflow-hidden hover:shadow-lg transition-shadow';
                
                const duration = formatDuration(track.duration_sec);
                const categoryName = track.category ? track.category.name : 'Unknown';
                const categoryColor = getCategoryColor(categoryName);
                
                div.innerHTML = `
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-4">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${categoryColor}">
                                ${categoryName}
                            </span>
                            <span class="text-sm text-gray-500">${duration}</span>
                        </div>
                        <h3 class="text-lg font-semibold text-gray-900 mb-2">${track.title}</h3>
                        <p class="text-gray-600 text-sm mb-4">${track.description || 'No description available.'}</p>
                        <div class="flex items-center justify-between">
                            <div class="flex space-x-2">
                                ${track.tags ? track.tags.slice(0, 3).map(tag => 
                                    `<span class="inline-flex items-center px-2 py-1 rounded text-xs font-medium bg-gray-100 text-gray-800">${tag}</span>`
                                ).join('') : ''}
                            </div>
                            <a href="/library/${track.id}" 
                               class="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700">
                                Listen
                            </a>
                        </div>
                    </div>
                `;
                
                return div;
            }

            function getCategoryColor(categoryName) {
                const colors = {
                    'Sleep': 'bg-indigo-100 text-indigo-800',
                    'Stress': 'bg-green-100 text-green-800',
                    'Confidence': 'bg-yellow-100 text-yellow-800',
                    'Focus': 'bg-purple-100 text-purple-800',
                    'Meditation': 'bg-blue-100 text-blue-800'
                };
                return colors[categoryName] || 'bg-gray-100 text-gray-800';
            }

            function formatDuration(seconds) {
                if (!seconds) return '0:00';
                const mins = Math.floor(seconds / 60);
                const secs = Math.floor(seconds % 60);
                return `${mins}:${secs.toString().padStart(2, '0')}`;
            }

            function handleSearch() {
                currentPage = 1;
                loadTracks(1, false);
            }

            function handleFilterChange() {
                currentPage = 1;
                loadTracks(1, false);
            }

            function showLoadingState(append = false) {
                if (!append) {
                    const container = document.getElementById('tracks-container');
                    container.innerHTML = `
                        <div class="col-span-full text-center py-12">
                            <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600 mx-auto"></div>
                            <p class="mt-4 text-gray-500">Loading tracks...</p>
                        </div>
                    `;
                }
            }

            function hideLoadingState() {
                // Loading state is replaced by content
            }

            function showEmptyState() {
                document.getElementById('empty-state').classList.remove('hidden');
            }

            function hideEmptyState() {
                document.getElementById('empty-state').classList.add('hidden');
            }

            function showError(message) {
                const container = document.getElementById('tracks-container');
                container.innerHTML = `
                    <div class="col-span-full text-center py-12">
                        <svg class="mx-auto h-12 w-12 text-red-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                        </svg>
                        <h3 class="mt-2 text-sm font-medium text-gray-900">Error</h3>
                        <p class="mt-1 text-sm text-gray-500">${message}</p>
                    </div>
                `;
            }

            function debounce(func, wait) {
                let timeout;
                return function executedFunction(...args) {
                    const later = () => {
                        clearTimeout(timeout);
                        func(...args);
                    };
                    clearTimeout(timeout);
                    timeout = setTimeout(later, wait);
                };
            }
        });
    </script>
</x-app-layout>
