<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ClassificationRule extends Model
{
    protected $fillable = [
        'pattern',
        'scope',
        'department_id',
        'account_id',
        'is_personal',
        'weight',
        'user_id',
        'usage_count',
        'last_used_at',
    ];

    protected $casts = [
        'is_personal' => 'boolean',
        'last_used_at' => 'datetime',
    ];

    public function department(): BelongsTo
    {
        return $this->belongsTo(Department::class);
    }

    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Increment usage count and update last used timestamp
     */
    public function recordUsage(): void
    {
        $this->increment('usage_count');
        $this->update(['last_used_at' => now()]);
    }

    /**
     * Get user-specific rules ordered by weight and usage
     */
    public static function getUserRules(int $userId): \Illuminate\Database\Eloquent\Collection
    {
        return static::where('user_id', $userId)
            ->orderByDesc('weight')
            ->orderByDesc('usage_count')
            ->get();
    }

    /**
     * Get global rules (no user_id) ordered by weight
     */
    public static function getGlobalRules(): \Illuminate\Database\Eloquent\Collection
    {
        return static::whereNull('user_id')
            ->orderByDesc('weight')
            ->get();
    }
}
