<?php

namespace App\Livewire\Admin;

use App\Models\Order;
use App\Models\User;
use Livewire\Component;
use Livewire\WithPagination;

class Orders extends Component
{
    use WithPagination;
    
    public $search = '';
    public $statusFilter = '';
    public $dateFilter = '';
    public $customerFilter = '';
    public $perPage = 20;
    
    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'dateFilter' => ['except' => ''],
        'customerFilter' => ['except' => ''],
        'perPage' => ['except' => 20],
    ];
    
    public function updatingSearch()
    {
        $this->resetPage();
    }
    
    public function updatingStatusFilter()
    {
        $this->resetPage();
    }
    
    public function updatingDateFilter()
    {
        $this->resetPage();
    }
    
    public function updatingCustomerFilter()
    {
        $this->resetPage();
    }

    public function viewOrder($orderId)
    {
        return redirect()->route('admin.orders.show', $orderId);
    }

    public function deleteOrder($orderId)
    {
        $order = Order::findOrFail($orderId);
        
        // Check if order is recent (within last 30 days) and not cancelled
        $thirtyDaysAgo = now()->subDays(30);
        
        if ($order->created_at >= $thirtyDaysAgo && $order->status !== 'cancelled') {
            session()->flash('error', "Cannot delete recent order #{$order->order_number} that is not cancelled. Please cancel the order first or wait 30 days.");
            return;
        }
        
        $orderNumber = $order->order_number;
        
        // Delete order lines first
        $order->orderLines()->delete();
        
        // Delete the order
        $order->delete();
        
        session()->flash('message', "Order #{$orderNumber} has been deleted successfully!");
    }
    
    public function render()
    {
        $query = Order::with(['user'])
            ->withCount('orderLines');

        // Search filter
        if ($this->search) {
            $query->where(function ($q) {
                $q->where('order_number', 'like', '%' . $this->search . '%')
                  ->orWhere('zoho_invoice_number', 'like', '%' . $this->search . '%')
                  ->orWhere('supplier_invoice_number', 'like', '%' . $this->search . '%')
                  ->orWhere('tracking_number', 'like', '%' . $this->search . '%')
                  ->orWhereHas('user', function ($q) {
                      $q->where('name', 'like', '%' . $this->search . '%')
                        ->orWhere('email', 'like', '%' . $this->search . '%');
                  });
            });
        }

        // Status filter
        if ($this->statusFilter) {
            $query->where('status', $this->statusFilter);
        }

        // Date filter
        if ($this->dateFilter) {
            $now = now();
            switch ($this->dateFilter) {
                case 'today':
                    $query->whereDate('created_at', $now->toDateString());
                    break;
                case 'week':
                    $query->whereBetween('created_at', [
                        $now->startOfWeek()->toDateString(),
                        $now->endOfWeek()->toDateString()
                    ]);
                    break;
                case 'month':
                    $query->whereMonth('created_at', $now->month)
                          ->whereYear('created_at', $now->year);
                    break;
                case 'year':
                    $query->whereYear('created_at', $now->year);
                    break;
                case 'last_30_days':
                    $query->where('created_at', '>=', $now->subDays(30));
                    break;
            }
        }

        // Customer filter
        if ($this->customerFilter) {
            $query->where('user_id', $this->customerFilter);
        }

        $orders = $query->orderBy('created_at', 'desc')
                       ->paginate($this->perPage);

        // Get all customers for filter dropdown
        $customers = User::where('role', 'user')
                        ->orderBy('name')
                        ->get();

        // Calculate totals
        $totalOrders = Order::count();
        $totalRevenue = Order::sum('initial_total_amount');
        $pendingOrders = Order::where('status', 'pending')->count();
        $processingOrders = Order::whereIn('status', ['confirmed', 'processing'])->count();

        return view('livewire.admin.orders', [
            'orders' => $orders,
            'customers' => $customers,
            'totalOrders' => $totalOrders,
            'totalRevenue' => $totalRevenue,
            'pendingOrders' => $pendingOrders,
            'processingOrders' => $processingOrders,
        ])
            ->layout('layouts.admin')
            ->title('Orders');
    }
}
