<?php

namespace App\Jobs;

use App\Models\Receipt;
use App\Services\Matching\ReceiptMatchingService;
use App\Services\Matching\GroupedReceiptMatchingService;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessReceiptMatching implements ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $timeout = 60;
    public int $maxExceptions = 2;

    public function __construct(public int $receiptId) {}

    public function backoff(): array
    {
        return [15, 60, 300];
    }

    public function retryUntil(): \DateTimeInterface
    {
        return now()->addHours(6);
    }

    public function handle(
        ReceiptMatchingService $matching,
        GroupedReceiptMatchingService $groupedMatching
    ): void {
        try {
            $receipt = Receipt::with('lines')->find($this->receiptId);
            if (!$receipt) {
                return;
            }

            // Check if receipt is part of a group
            $group = $receipt->receiptGroup;
            
            if ($group) {
                // If receipt was grouped, match the entire group
                $groupedMatching->attemptMatchForGroup($group);
            } else {
                // If not grouped, match individual receipt
                $matching->attemptMatchForReceipt($receipt);
            }

        } catch (\Throwable $e) {
            \Log::warning('ProcessReceiptMatching: transient failure, will retry', [
                'receipt_id' => $this->receiptId,
                'error' => $e->getMessage(),
                'attempt' => method_exists($this, 'attempts') ? $this->attempts() : null,
            ]);
            $schedule = (array) $this->backoff();
            $index = max(0, min((int)($this->attempts() - 1), count($schedule) - 1));
            $this->release((int) $schedule[$index]);
        }
    }
}
