<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Meditation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class MeditationController extends Controller
{
    /**
     * Display a listing of meditations
     */
    public function index(Request $request)
    {
        $meditations = $request->user()->meditations()
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return response()->json([
            'success' => true,
            'data' => $meditations
        ]);
    }

    /**
     * Store a newly created meditation
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'duration' => 'required|integer|min:1|max:120',
            'type' => 'required|string|in:mindfulness,loving_kindness,body_scan,breathing,transcendental,zen',
            'difficulty' => 'required|string|in:beginner,intermediate,advanced',
            'audio_url' => 'nullable|url',
            'guided' => 'boolean',
            'background_sounds' => 'nullable|string',
            'custom_script' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $meditation = $request->user()->meditations()->create([
            'title' => $request->title,
            'description' => $request->description,
            'duration' => $request->duration,
            'type' => $request->type,
            'difficulty' => $request->difficulty,
            'audio_url' => $request->audio_url,
            'guided' => $request->guided ?? true,
            'background_sounds' => $request->background_sounds,
            'custom_script' => $request->custom_script,
            'status' => 'created',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Meditation created successfully',
            'data' => $meditation
        ], 201);
    }

    /**
     * Display the specified meditation
     */
    public function show(Meditation $meditation)
    {
        if ($meditation->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        return response()->json([
            'success' => true,
            'data' => $meditation
        ]);
    }

    /**
     * Update the specified meditation
     */
    public function update(Request $request, Meditation $meditation)
    {
        if ($meditation->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'duration' => 'sometimes|required|integer|min:1|max:120',
            'type' => 'sometimes|required|string|in:mindfulness,loving_kindness,body_scan,breathing,transcendental,zen',
            'difficulty' => 'sometimes|required|string|in:beginner,intermediate,advanced',
            'audio_url' => 'nullable|url',
            'guided' => 'boolean',
            'background_sounds' => 'nullable|string',
            'custom_script' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $meditation->update($request->only([
            'title', 'description', 'duration', 'type', 
            'difficulty', 'audio_url', 'guided', 'background_sounds', 'custom_script'
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Meditation updated successfully',
            'data' => $meditation
        ]);
    }

    /**
     * Remove the specified meditation
     */
    public function destroy(Meditation $meditation)
    {
        if ($meditation->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $meditation->delete();

        return response()->json([
            'success' => true,
            'message' => 'Meditation deleted successfully'
        ]);
    }

    /**
     * Start a meditation
     */
    public function startMeditation(Meditation $meditation)
    {
        if ($meditation->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $meditation->update([
            'status' => 'in_progress',
            'started_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Meditation started successfully',
            'data' => $meditation
        ]);
    }

    /**
     * Complete a meditation
     */
    public function completeMeditation(Request $request, Meditation $meditation)
    {
        if ($meditation->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'completion_percentage' => 'required|integer|min:0|max:100',
            'mood_before' => 'nullable|integer|min:1|max:10',
            'mood_after' => 'nullable|integer|min:1|max:10',
            'focus_level' => 'nullable|integer|min:1|max:10',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $meditation->update([
            'status' => 'completed',
            'completed_at' => now(),
            'completion_percentage' => $request->completion_percentage,
            'mood_before' => $request->mood_before,
            'mood_after' => $request->mood_after,
            'focus_level' => $request->focus_level,
            'notes' => $request->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Meditation completed successfully',
            'data' => $meditation
        ]);
    }
}
