<?php

namespace App\Jobs;

use App\Models\LibraryTrack;
use App\Services\MediaService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class GenerateSignedUrl implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $timeout = 60; // 1 minute
    public $tries = 3;

    /**
     * Create a new job instance.
     */
    public function __construct(
        protected LibraryTrack $track,
        protected int $expiresIn = 3600 // 1 hour default
    ) {}

    /**
     * Execute the job.
     */
    public function handle(MediaService $mediaService): void
    {
        try {
            Log::info('Starting signed URL generation', [
                'track_id' => $this->track->id,
                'title' => $this->track->title,
                'expires_in' => $this->expiresIn
            ]);

            // Check if track has a file
            if (!$this->track->file_path) {
                throw new \Exception('Track has no associated file');
            }

            // Generate signed URL using MediaService
            $signedUrl = $mediaService->generateSignedUrl(
                $this->track->file_path,
                $this->expiresIn
            );

            // Update track with signed URL and expiration
            $this->track->update([
                'signed_url' => $signedUrl,
                'signed_url_expires_at' => now()->addSeconds($this->expiresIn),
                'last_signed_url_generated_at' => now()
            ]);

            Log::info('Signed URL generation completed', [
                'track_id' => $this->track->id,
                'signed_url' => $signedUrl,
                'expires_at' => $this->track->signed_url_expires_at
            ]);

        } catch (\Exception $e) {
            Log::error('Signed URL generation failed', [
                'track_id' => $this->track->id,
                'error' => $e->getMessage()
            ]);

            // Update track with failure information
            $this->track->update([
                'signed_url' => null,
                'signed_url_expires_at' => null,
                'last_signed_url_error' => $e->getMessage(),
                'last_signed_url_error_at' => now()
            ]);

            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('Generate signed URL job failed', [
            'track_id' => $this->track->id,
            'error' => $exception->getMessage()
        ]);

        // Update track with failure information
        $this->track->update([
            'signed_url' => null,
            'signed_url_expires_at' => null,
            'last_signed_url_error' => $exception->getMessage(),
            'last_signed_url_error_at' => now()
        ]);
    }
}
