<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Profile;
use App\Models\Consent;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Models\SafetyEvent;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        // Create admin user
        $adminUser = User::create([
            'name' => 'Admin User',
            'email' => 'admin@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
            'email_verified_at' => now(),
        ]);

        // Create demo user
        $demoUser = User::factory()->create([
            'name' => 'Demo User',
            'email' => 'demo@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
            'email_verified_at' => now(),
        ]);

        // Create profile for demo user
        Profile::create([
            'user_id' => $demoUser->id,
            'wellness_goals' => ['Reduce stress', 'improve sleep', 'build confidence'],
            'voice_preference' => 'female',
            'sensitivity_level' => 'standard',
            'medical_notes' => 'No known medical conditions',
            'preferences' => ['morning_sessions' => true, 'gentle_guidance' => true],
        ]);

        // Create consents for demo user
        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'privacy_policy',
            'description' => 'Privacy Policy and Data Processing Consent',
            'version' => '1.0',
            'is_accepted' => true,
            'accepted_at' => now(),
        ]);

        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'terms_of_service',
            'description' => 'Terms of Service Agreement',
            'version' => '1.0',
            'is_accepted' => true,
            'accepted_at' => now(),
        ]);

        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'data_processing',
            'description' => 'Data Processing and Analytics Consent',
            'version' => '1.0',
            'is_accepted' => true,
            'accepted_at' => now(),
        ]);

        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'audio_content',
            'description' => 'Audio Content and Hypnotherapy Sessions Consent',
            'version' => '1.0',
            'is_accepted' => true,
            'accepted_at' => now()->subDays(30),
            'expires_at' => now()->addMonths(6),
        ]);

        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'custom_sessions',
            'description' => 'Personalized Session Creation and Processing Consent',
            'version' => '1.1',
            'is_accepted' => true,
            'accepted_at' => now()->subDays(15),
            'expires_at' => now()->addYear(),
        ]);

        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'marketing_communications',
            'description' => 'Marketing and Promotional Communications Consent',
            'version' => '1.0',
            'is_accepted' => false,
            'accepted_at' => null,
            'expires_at' => now()->addYear(),
        ]);

        // Create library categories
        $categories = [
            [
                'name' => 'Meditation & Mindfulness',
                'slug' => 'meditation-mindfulness',
                'description' => 'Guided meditation sessions for stress relief and mental clarity',
                'icon' => '🧘',
                'color' => 'primary',
                'sort_order' => 1,
            ],
            [
                'name' => 'Sleep & Relaxation',
                'slug' => 'sleep-relaxation',
                'description' => 'Soothing sessions to help you fall asleep and stay asleep',
                'icon' => '😴',
                'color' => 'accent',
                'sort_order' => 2,
            ],
            [
                'name' => 'Confidence & Self-Esteem',
                'slug' => 'confidence-self-esteem',
                'description' => 'Build your confidence and develop a positive self-image',
                'icon' => '💪',
                'color' => 'primary',
                'sort_order' => 3,
            ],
            [
                'name' => 'Stress & Anxiety Relief',
                'slug' => 'stress-anxiety-relief',
                'description' => 'Techniques and sessions to manage stress and anxiety',
                'icon' => '🌿',
                'color' => 'accent',
                'sort_order' => 4,
            ],
            [
                'name' => 'Focus & Productivity',
                'slug' => 'focus-productivity',
                'description' => 'Improve concentration and boost your productivity',
                'icon' => '🎯',
                'color' => 'primary',
                'sort_order' => 5,
            ],
            [
                'name' => 'Emotional Healing',
                'slug' => 'emotional-healing',
                'description' => 'Work through emotional challenges and find inner peace',
                'icon' => '❤️',
                'color' => 'accent',
                'sort_order' => 6,
            ],
        ];

        foreach ($categories as $categoryData) {
            LibraryCategory::create($categoryData);
        }

        // Create library tracks
        $tracks = [
            [
                'title' => 'Hypnotherapy Test Track',
                'slug' => 'hypnotherapy-test-track',
                'description' => 'A test track for hypnotherapy and audio playback functionality. This track helps verify the audio system is working correctly.',
                'category_id' => 1,
                'duration' => 62, // 1.02 minutes in seconds
                'file_path' => 'tracks/hypnoaudio.wav',
                'transcript' => 'This is a test track for the Vybe Web App audio system. The audio player should now be fully functional with this real audio file.',
                'tags' => ['test', 'hypnotherapy', 'audio', 'verification'],
                'voice_type' => 'neutral',
                'difficulty_level' => 'beginner',
                'is_active' => true,
                'play_count' => 0,
            ],

        ];

        foreach ($tracks as $trackData) {
            LibraryTrack::create($trackData);
        }

        // Create custom requests for demo user
        $customRequests = [
            [
                'user_id' => $demoUser->id,
                'topic' => 'Public Speaking Confidence',
                'description' => 'I have a big presentation coming up and I\'m terrified of public speaking. I need help building confidence and managing my nerves.',
                'session_length' => '30min',
                'voice_preference' => 'female',
                'special_notes' => 'I prefer gentle, encouraging guidance rather than aggressive motivation.',
                'status' => 'completed',
            ],
            [
                'user_id' => $demoUser->id,
                'topic' => 'Work-Life Balance',
                'description' => 'I\'m struggling to maintain boundaries between work and personal life. I need strategies to disconnect and recharge.',
                'session_length' => '45min',
                'voice_preference' => 'male',
                'special_notes' => 'I work from home, so I need practical strategies for creating separation.',
                'status' => 'in_progress',
            ],
        ];

        foreach ($customRequests as $requestData) {
            CustomRequest::create($requestData);
        }

        // Create custom sessions
        $customSessions = [
            [
                'user_id' => $demoUser->id,
                'custom_request_id' => 1,
                'title' => 'Public Speaking Confidence Builder',
                'description' => 'A personalized session to help you overcome public speaking anxiety and build unshakeable confidence.',
                'file_path' => null, // No audio file available yet
                'transcript' => 'Welcome to your personalized public speaking confidence session. I understand how challenging this can be...',
                'duration' => 1800, // 30 minutes in seconds
                'voice_type' => 'female',
                'status' => 'pending', // Changed from 'ready' since no audio file
                'generated_at' => null,
            ],
            [
                'user_id' => $demoUser->id,
                'custom_request_id' => 2,
                'title' => 'Work-Life Balance Mastery',
                'description' => 'Learn practical strategies to create healthy boundaries and maintain work-life harmony.',
                'file_path' => null, // No audio file available yet
                'transcript' => 'Finding balance between work and life is a journey many of us struggle with...',
                'duration' => 2700, // 45 minutes in seconds
                'voice_type' => 'male',
                'status' => 'pending', // Changed from 'generating' since no audio file
                'generated_at' => null,
            ],
        ];

        foreach ($customSessions as $sessionData) {
            CustomSession::create($sessionData);
        }

        // Create safety events
        $safetyEvents = [
            [
                'user_id' => $demoUser->id,
                'event_type' => 'content_warning',
                'description' => 'User reported feeling triggered by anxiety-related content',
                'severity' => 'low',
                'status' => 'resolved',
                'resolved_at' => now(),
            ],
            [
                'user_id' => $demoUser->id,
                'event_type' => 'technical_issue',
                'description' => 'Audio player occasionally stutters on mobile devices',
                'severity' => 'low',
                'status' => 'resolved',
                'resolution_notes' => 'Fixed by optimizing audio buffering for mobile devices',
                'resolved_at' => now()->subDays(5),
            ],
            [
                'user_id' => $demoUser->id,
                'event_type' => 'accessibility_concern',
                'description' => 'Audio controls need better keyboard navigation for screen readers',
                'severity' => 'medium',
                'status' => 'in_progress',
                'resolution_notes' => 'Working on implementing ARIA labels and keyboard shortcuts',
                'resolved_at' => null,
            ],
        ];

        foreach ($safetyEvents as $eventData) {
            SafetyEvent::create($eventData);
        }

        // Create additional users for variety
        User::factory(5)->create()->each(function ($user) {
            Profile::create([
                'user_id' => $user->id,
                'wellness_goals' => ['General wellness', 'stress management'],
                'voice_preference' => 'neutral',
                'sensitivity_level' => 'low',
                'medical_notes' => 'No known medical conditions',
                'preferences' => ['evening_sessions' => true],
            ]);

            Consent::create([
                'user_id' => $user->id,
                'type' => 'privacy_policy',
                'description' => 'Privacy Policy and Data Processing Consent',
                'version' => '1.0',
                'is_accepted' => true,
                'accepted_at' => now(),
            ]);
        });

        $this->command->info('Demo data seeded successfully!');
        $this->command->info('Demo user: demo@example.com / password');
    }
}
