<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Profile;
use App\Models\Consent;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Models\SafetyEvent;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        // Create demo user
        $demoUser = User::factory()->create([
            'name' => 'Demo User',
            'email' => 'demo@example.com',
            'password' => Hash::make('password'),
            'email_verified_at' => now(),
        ]);

        // Create profile for demo user
        Profile::create([
            'user_id' => $demoUser->id,
            'goals' => 'Reduce stress, improve sleep, build confidence',
            'voice_preferences' => 'female',
            'sensitivity_level' => 'medium',
            'timezone' => 'UTC',
        ]);

        // Create consents for demo user
        Consent::create([
            'user_id' => $demoUser->id,
            'data_processing' => true,
            'marketing_emails' => false,
            'third_party_sharing' => false,
            'session_recording' => true,
            'accepted_at' => now(),
        ]);

        // Create library categories
        $categories = [
            [
                'name' => 'Meditation & Mindfulness',
                'slug' => 'meditation-mindfulness',
                'description' => 'Guided meditation sessions for stress relief and mental clarity',
                'icon' => '🧘',
                'color' => 'primary',
            ],
            [
                'name' => 'Sleep & Relaxation',
                'slug' => 'sleep-relaxation',
                'description' => 'Soothing sessions to help you fall asleep and stay asleep',
                'icon' => '😴',
                'color' => 'accent',
            ],
            [
                'name' => 'Confidence & Self-Esteem',
                'slug' => 'confidence-self-esteem',
                'description' => 'Build your confidence and develop a positive self-image',
                'icon' => '💪',
                'color' => 'primary',
            ],
            [
                'name' => 'Stress & Anxiety Relief',
                'slug' => 'stress-anxiety-relief',
                'description' => 'Techniques and sessions to manage stress and anxiety',
                'icon' => '🌿',
                'color' => 'accent',
            ],
            [
                'name' => 'Focus & Productivity',
                'slug' => 'focus-productivity',
                'description' => 'Improve concentration and boost your productivity',
                'icon' => '🎯',
                'color' => 'primary',
            ],
            [
                'name' => 'Emotional Healing',
                'slug' => 'emotional-healing',
                'description' => 'Work through emotional challenges and find inner peace',
                'icon' => '❤️',
                'color' => 'accent',
            ],
        ];

        foreach ($categories as $categoryData) {
            LibraryCategory::create($categoryData);
        }

        // Create library tracks
        $tracks = [
            [
                'title' => 'Morning Mindfulness',
                'slug' => 'morning-mindfulness',
                'description' => 'Start your day with clarity and intention. This 10-minute session helps you set a positive tone for the day ahead.',
                'category_id' => 1,
                'duration' => 10,
                'file_path' => 'tracks/morning-mindfulness.mp3',
                'transcript' => 'Welcome to your morning mindfulness session. Find a comfortable position and take a deep breath...',
                'tags' => ['morning', 'mindfulness', 'intention', 'clarity'],
                'rating' => 4.8,
                'plays_count' => 1250,
            ],
            [
                'title' => 'Deep Sleep Journey',
                'slug' => 'deep-sleep-journey',
                'description' => 'A gentle journey into deep, restorative sleep. Perfect for those who struggle with insomnia or restless nights.',
                'category_id' => 2,
                'duration' => 20,
                'file_path' => 'tracks/deep-sleep-journey.mp3',
                'transcript' => 'As you settle into bed, let your body sink into the mattress. Feel the weight of your body...',
                'tags' => ['sleep', 'insomnia', 'relaxation', 'restorative'],
                'rating' => 4.9,
                'plays_count' => 2100,
            ],
            [
                'title' => 'Confidence Booster',
                'slug' => 'confidence-booster',
                'description' => 'Build unshakeable confidence through positive affirmations and empowering visualizations.',
                'category_id' => 3,
                'duration' => 15,
                'file_path' => 'tracks/confidence-booster.mp3',
                'transcript' => 'You are capable, you are worthy, you are enough. Let these words sink deep into your being...',
                'tags' => ['confidence', 'self-esteem', 'empowerment', 'affirmations'],
                'rating' => 4.7,
                'plays_count' => 890,
            ],
            [
                'title' => 'Anxiety Relief Protocol',
                'slug' => 'anxiety-relief-protocol',
                'description' => 'A comprehensive approach to managing anxiety with breathing techniques and calming guidance.',
                'category_id' => 4,
                'duration' => 25,
                'file_path' => 'tracks/anxiety-relief-protocol.mp3',
                'transcript' => 'Anxiety is a natural response, but we can learn to manage it. Let\'s start with your breath...',
                'tags' => ['anxiety', 'breathing', 'calm', 'management'],
                'rating' => 4.6,
                'plays_count' => 1560,
            ],
            [
                'title' => 'Focus Flow State',
                'slug' => 'focus-flow-state',
                'description' => 'Enter a state of deep focus and flow to maximize your productivity and creativity.',
                'category_id' => 5,
                'duration' => 30,
                'file_path' => 'tracks/focus-flow-state.mp3',
                'transcript' => 'Flow state is where magic happens. It\'s that sweet spot between challenge and skill...',
                'tags' => ['focus', 'productivity', 'flow', 'creativity'],
                'rating' => 4.5,
                'plays_count' => 720,
            ],
            [
                'title' => 'Heart Healing Session',
                'slug' => 'heart-healing-session',
                'description' => 'A gentle journey to heal emotional wounds and open your heart to love and compassion.',
                'category_id' => 6,
                'duration' => 45,
                'file_path' => 'tracks/heart-healing-session.mp3',
                'transcript' => 'Your heart has been through so much. It\'s time to give it the love and care it deserves...',
                'tags' => ['healing', 'emotional', 'love', 'compassion'],
                'rating' => 4.9,
                'plays_count' => 430,
            ],
        ];

        foreach ($tracks as $trackData) {
            LibraryTrack::create($trackData);
        }

        // Create custom requests for demo user
        $customRequests = [
            [
                'user_id' => $demoUser->id,
                'topic' => 'Public Speaking Confidence',
                'description' => 'I have a big presentation coming up and I\'m terrified of public speaking. I need help building confidence and managing my nerves.',
                'session_length' => '30min',
                'voice_preference' => 'female',
                'special_notes' => 'I prefer gentle, encouraging guidance rather than aggressive motivation.',
                'status' => 'completed',
            ],
            [
                'user_id' => $demoUser->id,
                'topic' => 'Work-Life Balance',
                'description' => 'I\'m struggling to maintain boundaries between work and personal life. I need strategies to disconnect and recharge.',
                'session_length' => '45min',
                'voice_preference' => 'male',
                'special_notes' => 'I work from home, so I need practical strategies for creating separation.',
                'status' => 'in_progress',
            ],
        ];

        foreach ($customRequests as $requestData) {
            CustomRequest::create($requestData);
        }

        // Create custom sessions
        $customSessions = [
            [
                'user_id' => $demoUser->id,
                'custom_request_id' => 1,
                'title' => 'Public Speaking Confidence Builder',
                'description' => 'A personalized session to help you overcome public speaking anxiety and build unshakeable confidence.',
                'file_path' => 'custom-sessions/public-speaking-confidence.mp3',
                'transcript' => 'Welcome to your personalized public speaking confidence session. I understand how challenging this can be...',
                'duration' => 30,
                'status' => 'ready',
                'generated_at' => now(),
            ],
            [
                'user_id' => $demoUser->id,
                'custom_request_id' => 2,
                'title' => 'Work-Life Balance Mastery',
                'description' => 'Learn practical strategies to create healthy boundaries and maintain work-life harmony.',
                'file_path' => 'custom-sessions/work-life-balance.mp3',
                'transcript' => 'Finding balance between work and life is a journey many of us struggle with...',
                'duration' => 45,
                'status' => 'generating',
                'generated_at' => null,
            ],
        ];

        foreach ($customSessions as $sessionData) {
            CustomSession::create($sessionData);
        }

        // Create safety events
        $safetyEvents = [
            [
                'user_id' => $demoUser->id,
                'event_type' => 'content_warning',
                'description' => 'User reported feeling triggered by anxiety-related content',
                'severity' => 'low',
                'resolved' => true,
                'resolved_at' => now(),
            ],
        ];

        foreach ($safetyEvents as $eventData) {
            SafetyEvent::create($eventData);
        }

        // Create additional users for variety
        User::factory(5)->create()->each(function ($user) {
            Profile::create([
                'user_id' => $user->id,
                'goals' => 'General wellness and stress management',
                'voice_preferences' => 'neutral',
                'sensitivity_level' => 'low',
                'timezone' => 'UTC',
            ]);

            Consent::create([
                'user_id' => $user->id,
                'data_processing' => true,
                'marketing_emails' => true,
                'third_party_sharing' => false,
                'session_recording' => true,
                'accepted_at' => now(),
            ]);
        });

        $this->command->info('Demo data seeded successfully!');
        $this->command->info('Demo user: demo@example.com / password');
    }
}
