<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Profile;
use App\Models\Consent;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Models\SafetyEvent;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        // Create demo user
        $demoUser = User::factory()->create([
            'name' => 'Demo User',
            'email' => 'demo@example.com',
            'password' => Hash::make('password'),
            'email_verified_at' => now(),
        ]);

        // Create profile for demo user
        Profile::create([
            'user_id' => $demoUser->id,
            'wellness_goals' => ['Reduce stress', 'improve sleep', 'build confidence'],
            'voice_preference' => 'female',
            'sensitivity_level' => 'standard',
            'medical_notes' => 'No known medical conditions',
            'preferences' => ['morning_sessions' => true, 'gentle_guidance' => true],
        ]);

        // Create consents for demo user
        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'privacy_policy',
            'description' => 'Privacy Policy and Data Processing Consent',
            'version' => '1.0',
            'is_accepted' => true,
            'accepted_at' => now(),
        ]);

        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'terms_of_service',
            'description' => 'Terms of Service Agreement',
            'version' => '1.0',
            'is_accepted' => true,
            'accepted_at' => now(),
        ]);

        Consent::create([
            'user_id' => $demoUser->id,
            'type' => 'data_processing',
            'description' => 'Data Processing and Analytics Consent',
            'version' => '1.0',
            'is_accepted' => true,
            'accepted_at' => now(),
        ]);

        // Create library categories
        $categories = [
            [
                'name' => 'Meditation & Mindfulness',
                'slug' => 'meditation-mindfulness',
                'description' => 'Guided meditation sessions for stress relief and mental clarity',
                'icon' => '🧘',
                'color' => 'primary',
                'sort_order' => 1,
            ],
            [
                'name' => 'Sleep & Relaxation',
                'slug' => 'sleep-relaxation',
                'description' => 'Soothing sessions to help you fall asleep and stay asleep',
                'icon' => '😴',
                'color' => 'accent',
                'sort_order' => 2,
            ],
            [
                'name' => 'Confidence & Self-Esteem',
                'slug' => 'confidence-self-esteem',
                'description' => 'Build your confidence and develop a positive self-image',
                'icon' => '💪',
                'color' => 'primary',
                'sort_order' => 3,
            ],
            [
                'name' => 'Stress & Anxiety Relief',
                'slug' => 'stress-anxiety-relief',
                'description' => 'Techniques and sessions to manage stress and anxiety',
                'icon' => '🌿',
                'color' => 'accent',
                'sort_order' => 4,
            ],
            [
                'name' => 'Focus & Productivity',
                'slug' => 'focus-productivity',
                'description' => 'Improve concentration and boost your productivity',
                'icon' => '🎯',
                'color' => 'primary',
                'sort_order' => 5,
            ],
            [
                'name' => 'Emotional Healing',
                'slug' => 'emotional-healing',
                'description' => 'Work through emotional challenges and find inner peace',
                'icon' => '❤️',
                'color' => 'accent',
                'sort_order' => 6,
            ],
        ];

        foreach ($categories as $categoryData) {
            LibraryCategory::create($categoryData);
        }

        // Create library tracks
        $tracks = [
            [
                'title' => 'Morning Mindfulness',
                'slug' => 'morning-mindfulness',
                'description' => 'Start your day with clarity and intention. This 10-minute session helps you set a positive tone for the day ahead.',
                'category_id' => 1,
                'duration' => 600, // 10 minutes in seconds
                'audio_path' => 'tracks/morning-mindfulness.mp3',
                'transcript' => 'Welcome to your morning mindfulness session. Find a comfortable position and take a deep breath...',
                'tags' => ['morning', 'mindfulness', 'intention', 'clarity'],
                'voice_type' => 'female',
                'difficulty' => 'beginner',
                'is_active' => true,
                'play_count' => 1250,
            ],
            [
                'title' => 'Deep Sleep Journey',
                'slug' => 'deep-sleep-journey',
                'description' => 'A gentle journey into deep, restorative sleep. Perfect for those who struggle with insomnia or restless nights.',
                'category_id' => 2,
                'duration' => 1200, // 20 minutes in seconds
                'audio_path' => 'tracks/deep-sleep-journey.mp3',
                'transcript' => 'As you settle into bed, let your body sink into the mattress. Feel the weight of your body...',
                'tags' => ['sleep', 'insomnia', 'relaxation', 'restorative'],
                'voice_type' => 'female',
                'difficulty' => 'beginner',
                'is_active' => true,
                'play_count' => 2100,
            ],
            [
                'title' => 'Confidence Booster',
                'slug' => 'confidence-booster',
                'description' => 'Build unshakeable confidence through positive affirmations and empowering visualizations.',
                'category_id' => 3,
                'duration' => 900, // 15 minutes in seconds
                'audio_path' => 'tracks/confidence-booster.mp3',
                'transcript' => 'You are capable, you are worthy, you are enough. Let these words sink deep into your being...',
                'tags' => ['confidence', 'self-esteem', 'empowerment', 'affirmations'],
                'voice_type' => 'male',
                'difficulty' => 'intermediate',
                'is_active' => true,
                'play_count' => 890,
            ],
            [
                'title' => 'Anxiety Relief Protocol',
                'slug' => 'anxiety-relief-protocol',
                'description' => 'A comprehensive approach to managing anxiety with breathing techniques and calming guidance.',
                'category_id' => 4,
                'duration' => 1500, // 25 minutes in seconds
                'audio_path' => 'tracks/anxiety-relief-protocol.mp3',
                'transcript' => 'Anxiety is a natural response, but we can learn to manage it. Let\'s start with your breath...',
                'tags' => ['anxiety', 'breathing', 'calm', 'management'],
                'voice_type' => 'neutral',
                'difficulty' => 'intermediate',
                'is_active' => true,
                'play_count' => 1560,
            ],
            [
                'title' => 'Focus Flow State',
                'slug' => 'focus-flow-state',
                'description' => 'Enter a state of deep focus and flow to maximize your productivity and creativity.',
                'category_id' => 5,
                'duration' => 1800, // 30 minutes in seconds
                'audio_path' => 'tracks/focus-flow-state.mp3',
                'transcript' => 'Flow state is where magic happens. It\'s that sweet spot between challenge and skill...',
                'tags' => ['focus', 'productivity', 'flow', 'creativity'],
                'voice_type' => 'male',
                'difficulty' => 'advanced',
                'is_active' => true,
                'play_count' => 720,
            ],
            [
                'title' => 'Heart Healing Session',
                'slug' => 'heart-healing-session',
                'description' => 'A gentle journey to heal emotional wounds and open your heart to love and compassion.',
                'category_id' => 6,
                'duration' => 2700, // 45 minutes in seconds
                'audio_path' => 'tracks/heart-healing-session.mp3',
                'transcript' => 'Your heart has been through so much. It\'s time to give it the love and care it deserves...',
                'tags' => ['healing', 'emotional', 'love', 'compassion'],
                'voice_type' => 'female',
                'difficulty' => 'intermediate',
                'is_active' => true,
                'play_count' => 430,
            ],
        ];

        foreach ($tracks as $trackData) {
            LibraryTrack::create($trackData);
        }

        // Create custom requests for demo user
        $customRequests = [
            [
                'user_id' => $demoUser->id,
                'topic' => 'Public Speaking Confidence',
                'description' => 'I have a big presentation coming up and I\'m terrified of public speaking. I need help building confidence and managing my nerves.',
                'session_length' => '30min',
                'voice_preference' => 'female',
                'special_notes' => 'I prefer gentle, encouraging guidance rather than aggressive motivation.',
                'status' => 'completed',
            ],
            [
                'user_id' => $demoUser->id,
                'topic' => 'Work-Life Balance',
                'description' => 'I\'m struggling to maintain boundaries between work and personal life. I need strategies to disconnect and recharge.',
                'session_length' => '45min',
                'voice_preference' => 'male',
                'special_notes' => 'I work from home, so I need practical strategies for creating separation.',
                'status' => 'in_progress',
            ],
        ];

        foreach ($customRequests as $requestData) {
            CustomRequest::create($requestData);
        }

        // Create custom sessions
        $customSessions = [
            [
                'user_id' => $demoUser->id,
                'custom_request_id' => 1,
                'title' => 'Public Speaking Confidence Builder',
                'description' => 'A personalized session to help you overcome public speaking anxiety and build unshakeable confidence.',
                'file_path' => 'custom-sessions/public-speaking-confidence.mp3',
                'transcript' => 'Welcome to your personalized public speaking confidence session. I understand how challenging this can be...',
                'duration' => 1800, // 30 minutes in seconds
                'voice_type' => 'female',
                'status' => 'ready',
                'generated_at' => now(),
            ],
            [
                'user_id' => $demoUser->id,
                'custom_request_id' => 2,
                'title' => 'Work-Life Balance Mastery',
                'description' => 'Learn practical strategies to create healthy boundaries and maintain work-life harmony.',
                'file_path' => 'custom-sessions/work-life-balance.mp3',
                'transcript' => 'Finding balance between work and life is a journey many of us struggle with...',
                'duration' => 2700, // 45 minutes in seconds
                'voice_type' => 'male',
                'status' => 'generating',
                'generated_at' => null,
            ],
        ];

        foreach ($customSessions as $sessionData) {
            CustomSession::create($sessionData);
        }

        // Create safety events
        $safetyEvents = [
            [
                'user_id' => $demoUser->id,
                'event_type' => 'content_warning',
                'description' => 'User reported feeling triggered by anxiety-related content',
                'severity' => 'low',
                'status' => 'resolved',
                'resolved_at' => now(),
            ],
        ];

        foreach ($safetyEvents as $eventData) {
            SafetyEvent::create($eventData);
        }

        // Create additional users for variety
        User::factory(5)->create()->each(function ($user) {
            Profile::create([
                'user_id' => $user->id,
                'wellness_goals' => ['General wellness', 'stress management'],
                'voice_preference' => 'neutral',
                'sensitivity_level' => 'low',
                'medical_notes' => 'No known medical conditions',
                'preferences' => ['evening_sessions' => true],
            ]);

            Consent::create([
                'user_id' => $user->id,
                'type' => 'privacy_policy',
                'description' => 'Privacy Policy and Data Processing Consent',
                'version' => '1.0',
                'is_accepted' => true,
                'accepted_at' => now(),
            ]);
        });

        $this->command->info('Demo data seeded successfully!');
        $this->command->info('Demo user: demo@example.com / password');
    }
}
