<?php

namespace App\Models;

use Laravel\Jetstream\HasTeams;
use Laravel\Sanctum\HasApiTokens;
use OwenIt\Auditing\Models\Audit;
use Laravel\Jetstream\HasProfilePhoto;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

/**
 * User model representing application users.
 * 
 * @package App\Models
 */
class User extends Authenticatable
{
    use HasFactory;
    use SoftDeletes;
    use HasApiTokens;
    use HasProfilePhoto;
    use HasTeams;
    use Notifiable;
    use TwoFactorAuthenticatable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'current_team_id',
        'deleted_at',
        'email',
        'email_verified_at',
        'name',
        'password',
        'phone',
        'profile_photo_path',
        'remember_token',
        'settings',
        'two_factor_confirmed_at',
        'two_factor_recovery_codes',
        'two_factor_secret',
        'zoho_refresh_token',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'created_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
        'email_verified_at' => 'datetime',
        'settings' => 'array',
        'updated_at' => 'datetime:Y-m-d',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array<int, string>
     */
    protected $appends = [
        'profile_photo_url',
    ];

    /**
     * Get the latest audit record for this user.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function latestAudit()
    {
        return $this->hasOne(Audit::class)->latestOfMany();
    }

    /**
     * Get the teams that the user belongs to.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function teams()
    {
        return $this->belongsToMany(Team::class)->withPivot('role');
    }

    /**
     * Get the user departments.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function userDepartments()
    {
        return $this->hasMany(UserDepartment::class, 'users_id');
    }

    /**
     * Retrieve a setting with a given name or fall back to the default.
     * 
     * @param string $name The setting name
     * @param mixed $default The default value
     * @return mixed
     */
    public function setting(string $name, $default = null)
    {
        if ($this->settings !== null && array_key_exists($name, $this->settings)) {
            return $this->settings[$name];
        }
        
        return $default;
    }

    /**
     * Update one or more settings and then optionally save the model.
     * 
     * @param array $revisions The settings to update
     * @param bool $save Whether to save the model
     * @return self
     */
    public function settings(array $revisions, bool $save = true): self
    {
        if ($this->settings !== null) {
            $this->settings = array_merge($this->settings, $revisions);
        } else {
            $this->settings = $revisions;
        }
        
        if ($save) {
            $this->save();
        }
        
        return $this;
    }
}
