<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Services\AI\ReceiptDeduplicationService;
use Illuminate\Console\Command;

class TestSplitReceiptDetection extends Command
{
    protected $signature = 'test:split-receipt-detection';
    protected $description = 'Test split receipt detection for known patterns like Costco';

    public function handle()
    {
        $this->info('Testing Split Receipt Detection...');
        
        // Look for receipts with Costco-related merchant names
        $costcoReceipts = Receipt::where(function($query) {
            $query->where('merchant_name', 'LIKE', '%costco%')
                  ->orWhere('merchant_name', 'LIKE', '%robert todd%')
                  ->orWhere('merchant_name', 'LIKE', '%todd and son%');
        })
        ->whereNull('receipt_group_id')
        ->whereDoesntHave('matches')
        ->with(['lines'])
        ->orderBy('receipt_date', 'desc')
        ->take(10)
        ->get();

        if ($costcoReceipts->count() < 2) {
            $this->warn('Not enough Costco-related receipts to test. Need at least 2 receipts.');
            $this->info('Looking for receipts with merchant names containing: costco, robert todd, todd and son');
            return;
        }

        $this->info("Found {$costcoReceipts->count()} Costco-related receipts:");
        
        foreach ($costcoReceipts as $receipt) {
            $this->line("  ID: {$receipt->id} | {$receipt->merchant_name} | £{$receipt->total_amount} | {$receipt->receipt_date}");
        }

        $this->info("\nTesting deduplication service...");
        
        $deduplicationService = app(ReceiptDeduplicationService::class);
        $results = [];
        
        foreach ($costcoReceipts as $receipt) {
            $this->info("Processing receipt {$receipt->id} ({$receipt->merchant_name})...");
            
            $group = $deduplicationService->processReceipt($receipt);
            
            if ($group) {
                $this->info("  ✅ Grouped! Group ID: {$group->id} - {$group->name}");
                $results[] = [
                    'receipt_id' => $receipt->id,
                    'group_id' => $group->id,
                    'group_name' => $group->name
                ];
            } else {
                $this->warn("  ❌ No grouping found");
            }
        }

        if (!empty($results)) {
            $this->info("\nGrouping Results:");
            foreach ($results as $result) {
                $this->line("  Receipt {$result['receipt_id']} → Group {$result['group_id']} ({$result['group_name']})");
            }
        } else {
            $this->warn("No groupings were created. This might indicate:");
            $this->line("  - Receipts are too far apart in date");
            $this->line("  - Amounts don't complement each other");
            $this->line("  - Pattern not recognized");
        }

        // Show existing groups
        $existingGroups = \App\Models\ReceiptGroup::with(['receipts'])
            ->whereHas('receipts', function($query) {
                $query->where(function($q) {
                    $q->where('merchant_name', 'LIKE', '%costco%')
                      ->orWhere('merchant_name', 'LIKE', '%robert todd%')
                      ->orWhere('merchant_name', 'LIKE', '%todd and son%');
                });
            })
            ->get();

        if ($existingGroups->count() > 0) {
            $this->info("\nExisting Costco-related groups:");
            foreach ($existingGroups as $group) {
                $this->line("  Group {$group->id}: {$group->name} ({$group->receipts->count()} receipts)");
                foreach ($group->receipts as $receipt) {
                    $this->line("    - Receipt {$receipt->id}: {$receipt->merchant_name} (£{$receipt->total_amount})");
                }
            }
        }

        $this->info("\nSplit receipt detection test completed!");
    }
}
