<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Hypno AI – Dev Console</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
  <style>
    :root { --bg:#0b1220; --panel:#111827; --muted:#9ca3af; --text:#e5e7eb; --brand:#22d3ee; --accent:#34d399; }
    * { box-sizing: border-box; }
    body { margin:0; font-family: Inter, system-ui, -apple-system, Segoe UI, Roboto, sans-serif; background: var(--bg); color: var(--text); }
    header { padding:24px; border-bottom:1px solid #111; background: linear-gradient(180deg, rgba(34,211,238,.06), transparent); }
    h1 { margin:0; font-size:20px; letter-spacing: .2px; }
    .container { display:grid; grid-template-columns: 360px 1fr; gap:16px; padding:16px; }
    .card { background: var(--panel); border:1px solid #1f2937; border-radius: 12px; overflow: hidden; }
    .card h2 { margin:0; padding:12px 14px; font-size:14px; color: var(--muted); border-bottom:1px solid #1f2937; }
    .card .body { padding:14px; }
    label { display:block; font-size:12px; color: var(--muted); margin:8px 0 6px; }
    input, select, textarea { width:100%; background:#0f172a; border:1px solid #334155; color:#e5e7eb; border-radius:8px; padding:10px 12px; font-size:14px; }
    textarea { min-height:80px; }
    .row { display:flex; gap:8px; align-items:center; }
    .btn { background: var(--brand); color:#001018; font-weight:600; border:none; border-radius:10px; padding:10px 14px; cursor:pointer; }
    .btn.alt { background: var(--accent); }
    .muted { color: var(--muted); font-size:12px; }
    .list { display:flex; flex-direction:column; gap:8px; }
    .item { border:1px solid #1f2937; border-radius:10px; padding:10px; }
    pre { background:#0b1020; border:1px solid #1f2937; border-radius:10px; padding:10px; overflow:auto; }
    code { font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, monospace; font-size:12px; }
    .pill { font-size:11px; padding:4px 8px; border-radius: 999px; background:#0b1020; border:1px solid #1f2937; color:#93c5fd; }
  </style>
</head>
<body>
  <header>
    <h1>Hypno AI – Dev Console</h1>
    <div class="muted">Local testing UI for auth, profile, library, and custom session flows</div>
  </header>
  <div class="container">
    <div class="card">
      <h2>Auth</h2>
      <div class="body">
        <label>Fake SIWA identity token (sub only)</label>
        <input id="siwaToken" placeholder="apple-sub-123 will be wrapped in a fake JWT" />
        <div class="row" style="margin-top:10px">
          <button class="btn" onclick="authApple()">Sign In with Apple</button>
          <button class="btn alt" onclick="refreshToken()">Refresh Token</button>
        </div>
        <div class="muted" style="margin-top:8px">Bearer token is stored in localStorage.</div>
      </div>
    </div>

    <div class="card">
      <h2>Me & Profile</h2>
      <div class="body">
        <div class="row">
          <button class="btn" onclick="getMe()">GET /v1/me</button>
          <button class="btn" onclick="getProfile()">GET /v1/profile</button>
        </div>
        <label style="margin-top:12px">Update Profile (JSON)</label>
        <textarea id="profileJson">{"display_name":"Tester","goals":["sleep"],"voice_prefs":{"voice":"female_british_soft"}}</textarea>
        <button class="btn" style="margin-top:8px" onclick="patchProfile()">PATCH /v1/profile</button>
      </div>
    </div>

    <div class="card">
      <h2>Library</h2>
      <div class="body">
        <div class="row">
          <button class="btn" onclick="listCategories()">GET /v1/library/categories</button>
          <button class="btn" onclick="listTracks()">GET /v1/library/tracks</button>
        </div>
        <label style="margin-top:12px">Track ID</label>
        <input id="trackId" placeholder="1" />
        <button class="btn" style="margin-top:8px" onclick="showTrack()">GET /v1/library/tracks/{id}</button>
      </div>
    </div>

    <div class="card">
      <h2>Custom</h2>
      <div class="body">
        <label>Overrides JSON</label>
        <textarea id="overridesJson">{"topic":"sleep","length_min":15,"voice":"female_british_soft"}</textarea>
        <div class="row" style="margin-top:8px">
          <button class="btn" onclick="postCustomRequest()">POST /v1/custom/requests</button>
          <button class="btn" onclick="listSessions()">GET /v1/custom/sessions</button>
        </div>
      </div>
    </div>

    <div class="card" style="grid-column: 1 / span 2">
      <h2>Output</h2>
      <div class="body">
        <div class="row" style="gap:6px; margin-bottom:8px">
          <span class="pill" id="env">{{ app()->environment() }}</span>
          <span class="pill" id="tokenState">No Token</span>
        </div>
        <pre id="out"><code>// Responses will appear here…</code></pre>
      </div>
    </div>
  </div>

<script>
const api = (path, opts={}) => {
  const base = '/api';
  const headers = Object.assign({ 'Content-Type': 'application/json' }, opts.headers || {});
  const token = localStorage.getItem('token');
  if (token) headers['Authorization'] = 'Bearer ' + token;
  return fetch(base + path, Object.assign({}, opts, { headers }));
};

const setOut = async (resp) => {
  try {
    const text = typeof resp === 'string' ? resp : await resp.text();
    document.getElementById('out').innerText = text;
  } catch (e) {
    document.getElementById('out').innerText = String(resp);
  }
};

const setTokenState = () => {
  const t = localStorage.getItem('token');
  document.getElementById('tokenState').innerText = t ? 'Token set' : 'No Token';
};

function b64url(obj) {
  return btoa(JSON.stringify(obj)).replace(/=+$/,'').replace(/\+/g,'-').replace(/\//g,'_');
}

async function authApple() {
  const sub = document.getElementById('siwaToken').value || 'apple-sub-123';
  const fake = 'hdr.' + b64url({ sub }) + '.sig';
  const r = await api('/v1/auth/apple', { method: 'POST', body: JSON.stringify({ identity_token: fake }) });
  const j = await r.json();
  if (r.ok && j.token) { localStorage.setItem('token', j.token); setTokenState(); }
  await setOut(JSON.stringify(j, null, 2));
}

async function refreshToken() {
  const r = await api('/v1/auth/refresh', { method: 'POST' });
  const j = await r.json();
  if (r.ok && j.token) { localStorage.setItem('token', j.token); setTokenState(); }
  await setOut(JSON.stringify(j, null, 2));
}

async function getMe() {
  const r = await api('/v1/me');
  await setOut(await r.text());
}

async function getProfile() {
  const r = await api('/v1/profile');
  await setOut(await r.text());
}

async function patchProfile() {
  let body;
  try { body = JSON.parse(document.getElementById('profileJson').value || '{}'); } catch { body = {}; }
  const r = await api('/v1/profile', { method: 'PATCH', body: JSON.stringify(body) });
  await setOut(await r.text());
}

async function listCategories() {
  const r = await api('/v1/library/categories');
  await setOut(await r.text());
}

async function listTracks() {
  const r = await api('/v1/library/tracks');
  await setOut(await r.text());
}

async function showTrack() {
  const id = document.getElementById('trackId').value || '1';
  const r = await api('/v1/library/tracks/' + id);
  await setOut(await r.text());
}

async function postCustomRequest() {
  let overrides;
  try { overrides = JSON.parse(document.getElementById('overridesJson').value || '{}'); } catch { overrides = {}; }
  const r = await api('/v1/custom/requests', {
    method: 'POST',
    headers: { 'Idempotency-Key': crypto.randomUUID() },
    body: JSON.stringify({ overrides })
  });
  await setOut(await r.text());
}

async function listSessions() {
  const r = await api('/v1/custom/sessions');
  await setOut(await r.text());
}

setTokenState();
</script>
</body>
</html>


