<?php

namespace App\Models;

use App\Casts\Upper;
use \Bkwld\Cloner\Cloneable;
use App\Models\BaseModel;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

/**
 * Size model
 * 
 * Represents garment sizes used in the system.
 */
class Sizes extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'order',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'name' => Upper::class,
        'order' => 'integer',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
    ];

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) {
            $model->samples?->each?->delete();
            $model->customer_order_line_quantities?->each?->delete();
            $model->shipment_line_sizes?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->samples()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->customer_order_line_quantities()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->shipment_line_sizes()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('order');
        });
    }

    /**
     * Get the samples for this size.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function samples()
    {
        return $this->hasMany(Samples::class);
    }

    /**
     * Get the customer order line quantities for this size.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customer_order_line_quantities()
    {
        return $this->hasMany(CustomerOrderLineQuantities::class);
    }

    /**
     * Get the shipment line sizes for this size.
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function shipment_line_sizes()
    {
        return $this->hasMany(ShipmentLineSizes::class);
    }

    /**
     * Scope a query to search for sizes.
     * 
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $value
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, string $value)
    {
        return $query->where('name', 'like', "%{$value}%");
    }
}
