# Vybe Photo Superuser Portal Guide

## Overview

The Superuser Portal provides comprehensive platform administration capabilities for managing users, subscription tiers, and monitoring platform health.

## Features

### 1. Dashboard
- **User Statistics**: Total users, active subscriptions, trials, and user growth
- **Revenue Analytics**: Monthly Recurring Revenue (MRR), Annual Recurring Revenue (ARR), and revenue breakdown by tier
- **Platform Metrics**: Total photos, storage usage, and user engagement
- **Activity Feed**: Recent administrative actions and audit trail
- **Issue Monitoring**: Past due accounts and suspended users

### 2. User Management
- **User Directory**: Searchable, filterable list of all photographers
- **Advanced Filtering**: Filter by tier, status, and sort by various criteria
- **User Profiles**: Detailed view of user information, subscription details, and usage statistics
- **User Actions**:
  - Update subscription tier
  - Adjust photo and storage limits (override tier defaults)
  - Suspend/activate accounts
  - Promote users to superuser
  - Delete user accounts (with confirmation)

### 3. Subscription Tier Management
- **Tier Configuration**: Create and manage subscription tiers
- **Dynamic Limits**: Set photo limits, storage limits, and pricing
- **Feature Management**: Define tier features and capabilities
- **Custom Domain Control**: Enable/disable custom domain support per tier
- **Tier Activation**: Activate or deactivate tiers as needed

### 4. Audit Log
- **Complete Audit Trail**: Track all administrative actions
- **Action Types**:
  - User created/updated/deleted/suspended/activated
  - Tier created/updated/deleted/assigned
  - Subscription modifications
  - Limit adjustments
  - Settings changes
  - Custom domain verifications
- **Detailed Logging**: Includes admin user, target user, action details, IP address, and timestamp

## Setup

### Prerequisites
- PostgreSQL database running
- Node.js application with dependencies installed
- Environment variables configured in `.env`

### Installation Steps

1. **Run the database migration**:
```bash
psql -h localhost -U postgres -d photowebsite -f migrations/002-add-superuser-support.sql
```

2. **Create your first superuser**:
```bash
./setup-superuser.sh
```

The script will:
- Apply the superuser migration
- Prompt for username, email, and password
- Create a superuser account with full privileges
- Configure the account with business tier limits

3. **Start the server**:
```bash
node src/server-multi-tenant.js
```

4. **Access the superuser portal**:
- Navigate to `http://app.localhost:3000/login` (or your configured app subdomain)
- Log in with your superuser credentials
- Access the portal at `http://app.localhost:3000/superuser`

## Manual Superuser Creation

If you prefer to create a superuser manually via SQL:

```sql
-- First, insert or update the user
INSERT INTO users (username, email, password_hash, full_name, subscription_tier, subscription_status, is_superuser)
VALUES ('admin', 'admin@example.com', '$2b$10$...', 'Administrator', 'business', 'active', TRUE)
ON CONFLICT (email) DO UPDATE SET is_superuser = TRUE;

-- Or update an existing user
UPDATE users SET is_superuser = TRUE WHERE email = 'admin@example.com';
```

## Accessing the Portal

The superuser portal is only accessible:
1. On the app subdomain (e.g., `app.vybephoto.com` or `app.localhost:3000`)
2. When logged in as a user with `is_superuser = TRUE`
3. All actions are logged to the `admin_audit_log` table

### Portal URLs

- Dashboard: `/superuser`
- User Management: `/superuser/users`
- User Details: `/superuser/users/:id`
- Tier Management: `/superuser/tiers`
- Create Tier: `/superuser/tiers/new`
- Edit Tier: `/superuser/tiers/:id/edit`
- Audit Log: `/superuser/audit-log`

## User Management

### Viewing Users

The user management page provides:
- **Search**: Search by username, email, or full name
- **Filters**: Filter by subscription tier and status
- **Sorting**: Sort by creation date, username, email, tier, photo count, or last login
- **Pagination**: Browse through users 50 at a time

### Managing Individual Users

For each user, you can:

**Update Subscription Tier**:
- Changes the user's subscription tier
- Automatically updates photo and storage limits based on tier
- Logs the action to audit log

**Adjust Custom Limits**:
- Override the default tier limits
- Set custom photo count limits
- Set custom storage limits (in GB)
- Useful for special cases or trial users

**Suspend Account**:
- Temporarily disable user access
- Sets `subscription_status` to 'suspended'
- User cannot log in or access their gallery
- Optionally add a reason for suspension

**Activate Account**:
- Re-enable a suspended account
- Restores appropriate subscription status

**Delete Account**:
- Permanently delete user and all their data
- Requires username confirmation
- Cannot delete other superusers
- Cascades to delete albums and photos

**Promote to Superuser**:
- Grant superuser privileges to a user
- Allows access to the superuser portal

## Tier Management

### Creating Tiers

When creating a new tier:

1. **Basic Information**:
   - Tier Key: Unique identifier (lowercase, underscores) - cannot be changed
   - Tier Name: Display name shown to users
   - Description: Brief description for marketing
   - Price: Monthly price in pounds (£)
   - Display Order: Sort order on pricing page

2. **Limits & Capabilities**:
   - Max Photos: Maximum number of photos
   - Max Storage: Maximum storage in GB
   - Allow Custom Domain: Enable custom domain support
   - Allow Multiple Domains: Enable multiple custom domains
   - Active: Whether tier is visible to users

3. **Features**:
   - Add feature descriptions
   - Shown on pricing page
   - Use to highlight tier benefits

### Editing Tiers

All tier properties can be updated except the tier key. Changes to limits will affect:
- New users signing up for the tier
- Existing users when their tier is manually updated
- The trigger automatically applies new limits

### Deactivating Tiers

Deactivating a tier:
- Hides it from new signups
- Existing users remain on the tier
- Can be reactivated later

## Security & Best Practices

### Superuser Security

1. **Limit Superuser Accounts**: Only create superuser accounts for trusted administrators
2. **Use Strong Passwords**: Superuser accounts have access to all platform data
3. **Monitor Audit Log**: Regularly review administrative actions
4. **Revoke When Needed**: Remove superuser status when no longer needed

### Audit Trail

All superuser actions are logged with:
- Admin user who performed the action
- Target user (if applicable)
- Action type and details
- IP address and user agent
- Timestamp

This provides accountability and helps track changes.

### User Privacy

When accessing user data:
- Respect user privacy
- Only access accounts when necessary (support, investigation, etc.)
- Don't share user information outside the platform
- Follow applicable data protection regulations (GDPR, etc.)

## Common Tasks

### Upgrading a User's Plan

1. Go to `/superuser/users`
2. Search for the user
3. Click "Edit" on their row
4. Use the "Change Subscription Tier" dropdown
5. Select new tier and click "Update Tier"

### Handling a Past Due Account

1. Review the user's status in `/superuser/users`
2. Check their Stripe subscription status
3. Contact user or suspend account if needed
4. Reactivate once payment is resolved

### Creating a Custom Pricing Tier

1. Go to `/superuser/tiers`
2. Click "Create New Tier"
3. Fill in tier details:
   - Set appropriate limits
   - Configure domain support
   - Add feature descriptions
4. Click "Create Tier"
5. Tier is now available for assignment

### Investigating User Issues

1. Go to `/superuser/users/:id` for the user
2. Review their usage statistics
3. Check subscription status
4. Review recent admin actions in audit log
5. Adjust limits or tier if needed

### Monitoring Platform Health

1. Check the dashboard regularly
2. Monitor MRR/ARR trends
3. Track user growth (new users last 30 days)
4. Watch for issues (past due, suspended accounts)
5. Review engagement (active last 7 days)

## Technical Details

### Database Tables

**users**:
- Added `is_superuser` column (boolean)
- Added `is_suspended` column (boolean)

**subscription_tiers**:
- New table for managing subscription tiers
- Stores pricing, limits, features
- Used by triggers to set user limits

**admin_audit_log**:
- New table for tracking admin actions
- Includes admin user, target, action type, details
- Indexed for performance

### Middleware

**requireSuperuser**:
- Checks `req.user.is_superuser === true`
- Redirects to login if not authenticated
- Returns 403 if not a superuser

**requireSuperuserSubdomain**:
- Ensures superuser portal only accessible on app subdomain
- Prevents access from user galleries

### Database Functions

**apply_subscription_limits()**:
- Trigger function on users table
- Automatically sets limits when tier changes
- Looks up limits from subscription_tiers table
- Falls back to hardcoded values if needed

**update_subscription_tiers_updated_at()**:
- Trigger function on subscription_tiers table
- Automatically updates `updated_at` timestamp

## Troubleshooting

### Cannot Access Superuser Portal

**Issue**: "Access Denied" or redirect to login

**Solutions**:
- Verify you're logged in
- Check `is_superuser` is TRUE in database:
  ```sql
  SELECT id, username, email, is_superuser FROM users WHERE email = 'your@email.com';
  ```
- Ensure you're on app subdomain (app.localhost or app.vybephoto.com)

### Migration Errors

**Issue**: Migration fails to run

**Solutions**:
- Check PostgreSQL connection
- Verify database credentials in `.env`
- Run migration manually:
  ```bash
  psql -U postgres -d photowebsite -f migrations/002-add-superuser-support.sql
  ```
- Check for existing tables/columns that might conflict

### User Can't Be Deleted

**Issue**: Cannot delete user account

**Possible Causes**:
- User is a superuser (cannot delete superusers)
- Username confirmation doesn't match
- Database foreign key constraints

**Solutions**:
- Verify username is typed correctly
- Remove superuser status first if needed
- Check database logs for constraint violations

## API Reference

### Superuser Module Functions

The `src/superuser.js` module exports these functions:

**User Management**:
- `getAllUsers(options)`: Get paginated user list with filters
- `getUserStatistics()`: Get platform-wide user statistics
- `updateUserTier(adminUserId, targetUserId, tier, req)`: Change user's tier
- `suspendUser(adminUserId, targetUserId, reason, req)`: Suspend user account
- `activateUser(adminUserId, targetUserId, req)`: Activate suspended account
- `updateUserLimits(adminUserId, targetUserId, maxPhotos, maxStorageBytes, req)`: Set custom limits
- `deleteUser(adminUserId, targetUserId, req)`: Permanently delete user
- `makeSuperuser(adminUserId, targetUserId, req)`: Promote to superuser
- `removeSuperuser(adminUserId, targetUserId, req)`: Revoke superuser status

**Tier Management**:
- `getAllTiers()`: Get all subscription tiers
- `getTierById(tierId)`: Get single tier
- `createTier(adminUserId, tierData, req)`: Create new tier
- `updateTier(adminUserId, tierId, tierData, req)`: Update existing tier
- `deactivateTier(adminUserId, tierId, req)`: Deactivate tier

**Analytics**:
- `getRevenueStatistics()`: Calculate MRR, ARR, and breakdown by tier
- `getRecentAdminActions(limit)`: Get recent audit log entries

**Audit Logging**:
- `logAdminAction(adminUserId, actionType, targetUserId, actionDetails, req)`: Log an action

All functions that modify data automatically log to the audit trail.

## Future Enhancements

Potential features for future versions:

- **Email Notifications**: Send emails to users on suspension/tier changes
- **Bulk Operations**: Suspend/update multiple users at once
- **Advanced Analytics**: Charts, graphs, and trend analysis
- **Export Data**: Export user lists and reports to CSV
- **Impersonation**: Log in as user for support purposes
- **Scheduled Reports**: Email reports on platform health
- **Billing Integration**: Deeper Stripe integration and payment management
- **Custom Roles**: Multiple admin roles with different permissions
- **Activity Dashboard**: Real-time monitoring of user activity

## Support

For issues or questions:
1. Check this guide first
2. Review the audit log for clues
3. Check application logs: `console.error` messages
4. Verify database state with SQL queries
5. Contact the development team

## License & Credits

Part of Vybe Photo platform. For internal use only.

