<?php

namespace App\Http\Controllers;

use App\Services\CustomSessionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CustomSessionController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {
        $this->middleware('auth');
    }

    public function create()
    {
        return view('custom.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'topic' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'length' => 'sometimes|string|in:5min,15min,30min,45min,60min',
            'voice_preference' => 'sometimes|string|in:female,male,neutral',
            'special_notes' => 'sometimes|nullable|string|max:1000',
        ]);
        
        $user = Auth::user();
        
        $customRequest = $this->customSessionService->createRequest([
            'topic' => $request->topic,
            'description' => $request->description,
            'length' => $request->length,
            'voice_preference' => $request->voice_preference,
            'special_notes' => $request->special_notes,
        ], $user);
        
        // Process the request immediately for demo purposes
        $this->customSessionService->processCustomRequest($customRequest);
        
        return redirect()->route('custom.sessions.index')
            ->with('success', 'Your custom session request has been submitted and is being processed.');
    }

    public function index(Request $request)
    {
        $user = Auth::user();
        $filters = $request->only(['status']);
        
        $sessions = $this->customSessionService->getUserSessions($user, $filters);
        $requests = $this->customSessionService->getUserRequests($user, $filters);
        
        return view('custom.index', compact('sessions', 'requests', 'filters'));
    }

    public function show(int $id)
    {
        $user = Auth::user();
        $session = $this->customSessionService->getSession($id, $user);
        
        if (!$session) {
            abort(404);
        }
        
        return view('custom.show', compact('session'));
    }
}
