<?php

namespace App\Http\Controllers;

use App\Models\CustomRequest;
use App\Models\CustomSession;
use App\Services\CustomSessionService;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;

class CustomSessionController extends Controller
{
    public function __construct(
        private CustomSessionService $customSessionService
    ) {}

    public function create(): View
    {
        return view('custom.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'topic' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'session_length' => 'required|integer|min:300|max:3600', // 5 min to 1 hour
            'voice_preference' => 'required|string|in:male,female,neutral',
            'special_notes' => 'nullable|string|max:1000',
        ]);

        $customRequest = CustomRequest::create([
            'user_id' => Auth::id(),
            'topic' => $request->topic,
            'description' => $request->description,
            'session_length' => $request->session_length,
            'voice_preference' => $request->voice_preference,
            'special_notes' => $request->special_notes,
            'status' => 'pending',
        ]);

        // Queue the session generation job
        $this->customSessionService->queueSessionGeneration($customRequest);

        return redirect()->route('custom.sessions.index')
            ->with('success', 'Custom session request submitted successfully! We\'ll notify you when it\'s ready.');
    }

    public function index(): View
    {
        $sessions = CustomSession::where('user_id', Auth::id())
            ->with('customRequest')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('custom.sessions.index', compact('sessions'));
    }

    public function show(CustomSession $session): View
    {
        // Ensure user can only view their own sessions
        if ($session->user_id !== Auth::id()) {
            abort(403, 'Unauthorized');
        }

        $session->load('customRequest');

        return view('custom.sessions.show', compact('session'));
    }
}
