const db = require('./src/db-multi-tenant');

async function fixData() {
  try {
    console.log('Checking existing subscription_status values...\n');
    
    // Check what values exist
    const statusCheck = await db.pool.query(`
      SELECT subscription_status, COUNT(*) as count
      FROM users
      GROUP BY subscription_status
      ORDER BY count DESC;
    `);
    
    console.log('Current subscription_status values:');
    statusCheck.rows.forEach(row => {
      console.log(`  ${row.subscription_status || '(null)'}: ${row.count} users`);
    });
    console.log('');
    
    // Update any NULL or invalid values to 'trialing'
    console.log('Updating invalid/null values to "trialing"...');
    const updateResult = await db.pool.query(`
      UPDATE users 
      SET subscription_status = 'trialing'
      WHERE subscription_status IS NULL 
         OR subscription_status NOT IN ('trialing', 'active', 'past_due', 'canceled', 'paused', 'suspended', 'pending_payment', 'incomplete', 'incomplete_expired', 'unpaid');
    `);
    
    console.log(`Updated ${updateResult.rowCount} rows\n`);
    
    // Now try to add the constraint
    console.log('Adding updated constraint...');
    await db.pool.query(`
      ALTER TABLE users 
      ADD CONSTRAINT users_subscription_status_check 
      CHECK (subscription_status IN ('active', 'trialing', 'past_due', 'canceled', 'incomplete', 'incomplete_expired', 'unpaid', 'pending_payment', 'paused', 'suspended'));
    `);
    
    console.log('✅ Constraint added successfully!');
    console.log('\nValid subscription_status values:');
    console.log('  - trialing, active, past_due, canceled');
    console.log('  - incomplete, incomplete_expired, unpaid');
    console.log('  - pending_payment (NEW for Stripe Checkout)');
    console.log('  - paused, suspended');
    
  } catch (error) {
    console.error('Error:', error.message);
    console.error(error);
  } finally {
    process.exit(0);
  }
}

fixData();

