<?php

namespace App\Services;

use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Services\AI\VertexClient;
use Illuminate\Support\Facades\Log;

class EMLProcessingService
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Process EML file to extract receipt data
     */
    public function processEMLFile(string $emlPath): ?array
    {
        try {
            $emlContent = file_get_contents($emlPath);
            if (!$emlContent) {
                Log::warning('Could not read EML file', ['path' => $emlPath]);
                return null;
            }

            // Extract receipt data from EML content
            $receiptData = $this->extractReceiptFromEML($emlContent);
            
            if (empty($receiptData)) {
                Log::warning('No receipt data found in EML file', ['path' => $emlPath]);
                return null;
            }

            return $receiptData;
            
        } catch (\Exception $e) {
            Log::error('Error processing EML file', [
                'path' => $emlPath,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Extract receipt data from EML content using LLM
     */
    private function extractReceiptFromEML(string $emlContent): ?array
    {
        try {
            $model = config('vertex.models.ocr', 'gemini-2.5-flash');
            $system = 'You are an expert at extracting receipt data from email content. Extract ALL available information from the email content to help with future analysis and matching.

REQUIRED FIELDS:
- merchant_name: Business name
- receipt_date: Date in YYYY-MM-DD format  
- total_amount: Total amount as decimal (e.g., 12.50)
- currency: Currency code (e.g., GBP, USD, EUR)
- lines: Array of line items with description, quantity, unit_price, line_total

VAT/TAX FIELDS (CRITICAL FOR UK BUSINESSES):
- vat_number: VAT registration number (if visible)
- vat_rate: Standard VAT rate (e.g., 20% for UK standard rate)
- subtotal: Amount before VAT/tax
- vat_amount: Total VAT amount
- net_amount: Amount excluding VAT
- vat_breakdown: Object with different VAT rates and amounts (e.g., {"20%": 10.00, "5%": 2.50})
- is_vat_inclusive: Boolean - whether prices include VAT
- vat_exemption_notes: Any notes about VAT exemptions or zero-rated items

OPTIONAL FIELDS (extract if visible):
- receipt_number: Receipt/invoice number
- cashier_name: Cashier or staff member name
- store_location: Store address or location
- phone_number: Business phone number
- email: Business email
- website: Business website
- payment_method: Cash, card, etc.
- discount_amount: Any discounts applied
- tip_amount: Tip or service charge
- transaction_id: Transaction reference number
- loyalty_points: Loyalty points earned
- opening_hours: Store hours if visible
- return_policy: Return policy text if visible
- additional_notes: Any other relevant text

For line items, extract:
- description: Item description
- quantity: Number of items
- unit_price: Price per unit
- line_total: Total for this line
- category: Product category if identifiable (food, fuel, clothing, etc.)
- brand: Brand name if visible
- barcode: Barcode number if visible
- vat_rate: VAT rate for this line (e.g., 20%, 5%, 0%, "VAT FREE", "EXEMPT")
- vat_amount: VAT amount for this line
- net_amount: Amount excluding VAT for this line
- is_vat_inclusive: Whether this line price includes VAT
- vat_code: VAT code if shown (e.g., "S", "R", "Z", "E")
- vat_status: Status like "STANDARD", "REDUCED", "ZERO-RATED", "EXEMPT"

VAT EXTRACTION PRIORITY:
- Look for VAT registration numbers (usually 9 or 12 digits starting with GB)
- Extract any VAT breakdowns showing different rates (20%, 5%, 0%, etc.)
- Identify VAT-inclusive vs VAT-exclusive pricing
- Look for VAT codes (S=Standard, R=Reduced, Z=Zero-rated, E=Exempt)
- Extract subtotals, VAT amounts, and net amounts
- Pay attention to VAT-free or exempt items

Output strict JSON with all available fields. Be thorough - extract every piece of text and data visible in the email content, with special attention to VAT/tax information.';

            $prompt = [
                'email_content' => $emlContent,
                'instructions' => 'Extract all receipt data from this email content'
            ];

            $response = $this->vertex->generate($model, json_encode($prompt), $system, ['responseMimeType' => 'application/json']);
            
            if (!isset($response['json']) || !is_array($response['json'])) {
                Log::warning('LLM failed to extract EML content');
                return null;
            }

            return $response['json'];
            
        } catch (\Exception $e) {
            Log::error('Error extracting receipt from EML', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Create receipt from EML data
     */
    public function createReceiptFromEML(string $emlPath, array $receiptData): Receipt
    {
        $receipt = Receipt::create([
            'user_id' => auth()->id() ?? 1,
            'merchant_name' => $receiptData['merchant_name'] ?? 'Unknown Merchant',
            'receipt_date' => $receiptData['receipt_date'] ?? null,
            'total_amount' => $receiptData['total_amount'] ?? 0,
            'currency' => $receiptData['currency'] ?? 'GBP',
            'vat_amount' => $receiptData['vat_amount'] ?? null,
            'subtotal' => $receiptData['subtotal'] ?? null,
            'discount_amount' => $receiptData['discount_amount'] ?? null,
            'image_path' => $emlPath,
            'image_disk' => 'private',
            'meta' => $receiptData,
        ]);

        // Create receipt lines
        if (isset($receiptData['lines']) && is_array($receiptData['lines'])) {
            foreach ($receiptData['lines'] as $lineData) {
                ReceiptLine::create([
                    'receipt_id' => $receipt->id,
                    'description' => $lineData['description'] ?? '',
                    'quantity' => $lineData['quantity'] ?? 1,
                    'unit_price' => $lineData['unit_price'] ?? 0,
                    'line_total' => $lineData['line_total'] ?? 0,
                    'vat_rate' => $lineData['vat_rate'] ?? null,
                    'vat_amount' => $lineData['vat_amount'] ?? null,
                    'meta' => $lineData,
                ]);
            }
        }

        return $receipt;
    }
}
