<?php

namespace App\Livewire\Admin;

use App\Models\Item;
use App\Models\Season;
use App\Models\Category;
use App\Models\Colourway;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Symfony\Component\Process\Process;

class ItemDetail extends Component
{
    use WithPagination, WithFileUploads;
    
    public Item $item;
    public $editing = false;
    
    // Form fields for editing
    public $name;
    public $slug;
    public $description;
    public $shortDescription;
    public $seasonsId;
    public $categoriesId;
    public $price;
    public $sku;
    public $isActive = false;
    public $noColumns = 3;
    
    // Specifications management
    public $specifications = [];
    public $newSpecName = '';
    public $newSpecValue = '';
    
    // Colourways management
    public $newColourwayReference = '';
    public $newColourwayImage;
    public $colourwayPdf; // uploaded PDF for batch colour code extraction
    public $tempColourwayImage; // temporary file for drag/drop/paste uploads
    public $colourwayImages = []; // array of images indexed by colourway ID
    
    protected $rules = [
        'name' => 'required|string|max:255',
        'slug' => 'required|string|max:255|unique:items,slug',
        'description' => 'nullable|string',
        'shortDescription' => 'nullable|string',
        'seasonsId' => 'required|exists:seasons,id',
        'categoriesId' => 'required|exists:categories,id',
        'price' => 'nullable|numeric|min:0',
        'sku' => 'nullable|string|max:255|unique:items,sku',
        'isActive' => 'boolean',
        'noColumns' => 'required|integer|min:1|max:6',
    ];
    
    public function mount($itemId)
    {
        $this->item = Item::with(['season', 'category', 'colourways'])->findOrFail($itemId);
        $this->loadFormData();
        
        // Auto-enable editing for newly created items
        if ($this->item->name === 'New Item') {
            $this->editing = true;
        }
    }
    
    public function loadFormData()
    {
        $this->name = $this->item->name;
        $this->slug = $this->item->slug;
        $this->description = $this->item->description;
        $this->shortDescription = $this->item->short_description;
        $this->seasonsId = $this->item->seasons_id;
        $this->categoriesId = $this->item->categories_id;
        $this->price = $this->item->price;
        $this->sku = $this->item->sku;
        $this->isActive = (bool) $this->item->is_active;
        $this->noColumns = $this->item->no_columns ?? 3;
        $this->specifications = is_array($this->item->specifications) ? $this->item->specifications : [];
    }
    
    public function toggleEdit()
    {
        if ($this->editing) {
            $this->loadFormData(); // Reset form data
        }
        $this->editing = !$this->editing;
    }
    
    public function save()
    {
        try {
            // Ensure noColumns has a valid value
            if (empty($this->noColumns) || !is_numeric($this->noColumns)) {
                $this->noColumns = 3;
            }
            $this->noColumns = (int) max(1, min(6, $this->noColumns));
            
            // Ensure isActive is a boolean
            $this->isActive = (bool) $this->isActive;
            
            // Update validation rules to exclude current item from unique checks
            $this->rules['slug'] = 'required|string|max:255|unique:items,slug,' . $this->item->id;
            $this->rules['sku'] = 'nullable|string|max:255|unique:items,sku,' . $this->item->id;
            
            $this->validate();
            
            $this->item->update([
                'name' => $this->name,
                'slug' => $this->slug,
                'description' => $this->description,
                'short_description' => $this->shortDescription,
                'seasons_id' => $this->seasonsId,
                'categories_id' => $this->categoriesId,
                'price' => $this->price,
                'sku' => $this->sku,
                'is_active' => $this->isActive,
                'specifications' => $this->specifications,
                'no_columns' => $this->noColumns,
            ]);
            
            $this->editing = false;
            $this->item->refresh();
            session()->flash('message', 'Item updated successfully!');
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Validation errors are automatically handled by Livewire
            throw $e;
        } catch (\Exception $e) {
            session()->flash('error', 'Error saving item: ' . $e->getMessage());
            Log::error('Item save error: ' . $e->getMessage(), [
                'item_id' => $this->item->id,
                'exception' => $e
            ]);
        }
    }
    
    public function cancel()
    {
        $this->loadFormData();
        $this->editing = false;
    }
    
    // Specifications management
    public function addSpecification()
    {
        if ($this->newSpecName && $this->newSpecValue) {
            $this->specifications[] = [
                'name' => $this->newSpecName,
                'value' => $this->newSpecValue,
            ];
            $this->newSpecName = '';
            $this->newSpecValue = '';
        }
    }
    
    public function removeSpecification($index)
    {
        unset($this->specifications[$index]);
        $this->specifications = array_values($this->specifications);
    }
    
    // Colourways management
    public function addColourway()
    {
        $this->validate([
            'newColourwayReference' => 'required|string|max:255',
            'newColourwayImage' => 'nullable|image|max:2048',
        ]);

        $imagePath = null;
        if ($this->newColourwayImage) {
            $imagePath = $this->newColourwayImage->store('colourways', 'public');
        }

        Colourway::create([
            'items_id' => $this->item->id,
            'name' => $this->newColourwayReference,
            'reference' => $this->newColourwayReference,
            'image' => $imagePath,
            'sort_order' => $this->item->colourways()->count(),
        ]);

        $this->newColourwayReference = '';
        $this->newColourwayImage = null;

        // Refresh the item to get updated colourways
        $this->item->refresh();
        
        session()->flash('message', 'Colourway added successfully!');
    }

    /**
     * Handle uploaded PDF to extract colour codes in visual order and dd() them.
     */
    public function processColourwayPdf(): void
    {
        $this->validate([
            'colourwayPdf' => 'required|file|mimes:pdf|max:10240', // 10MB
        ]);

        // Store PDF to a private temp location (not publicly accessible)
        $uuid = (string) Str::uuid();
        $relativePdf = $this->colourwayPdf->storeAs('tmp/colourway', $uuid . '.pdf', 'local');
        $pdfPath = storage_path('app/private/' . $relativePdf);

        // Prepare Python execution
        $python = base_path('.venv/bin/python');
        if (!is_file($python)) {
            $which = trim((string) @shell_exec('command -v python3'));
            $python = $which !== '' ? $which : 'python3';
        }
        $script = base_path('extract_codes.py');
        $outPrefix = storage_path('app/private/tmp/colourway/' . $uuid);

        // Ensure out dir exists
        @mkdir(dirname($outPrefix), 0775, true);

        $process = new Process([
            $python,
            $script,
            $pdfPath,
            '--cols', 'auto',
            '--rows-per-page', 'auto',
            '--out-prefix', $outPrefix,
        ]);
        $process->setTimeout(120);

        try {
            $process->run();
            if (!$process->isSuccessful()) {
                throw new \RuntimeException($process->getErrorOutput() ?: $process->getOutput());
            }

            $txtPath = $outPrefix . '.codes.txt';
            if (!is_file($txtPath)) {
                throw new \RuntimeException('Extractor did not produce TXT output.');
            }
            $content = trim((string) @file_get_contents($txtPath));
            $parts = array_values(array_filter(array_map(function ($s) {
                return trim($s);
            }, explode(',', $content)), function ($s) {
                return $s !== '';
            }));

            // Delete all existing colourways for this item before creating new ones
            $this->item->colourways()->delete();

            // Get current season ID for reference
            $currentSeasonId = $this->item->seasons_id;

            // Create colourways for each extracted code (reference)
            $created = [];
            $sort = 0;
            foreach ($parts as $code) {
                $ref = strtoupper($code);
                // Only accept 6-digit codes (keep behavior consistent with extractor)
                if (!preg_match('/^\d{6}$/', $ref)) {
                    continue;
                }
                $sort++;

                // Look for existing colourway with same reference from previous seasons
                $existingColourway = Colourway::where('reference', $ref)
                    ->whereHas('item', function ($query) use ($currentSeasonId) {
                        $query->where('seasons_id', '!=', $currentSeasonId);
                    })
                    ->whereNotNull('image')
                    ->first();

                $imagePath = null;
                if ($existingColourway && $existingColourway->image) {
                    // Copy the image from the existing colourway
                    $sourcePath = storage_path('app/public/' . $existingColourway->image);
                    if (file_exists($sourcePath)) {
                        $newImagePath = 'colourways/' . $this->item->id . '_' . $ref . '_' . time() . '.png';
                        $destinationPath = storage_path('app/public/' . $newImagePath);
                        
                        // Ensure directory exists
                        $dir = dirname($destinationPath);
                        if (!is_dir($dir)) {
                            mkdir($dir, 0755, true);
                        }
                        
                        if (copy($sourcePath, $destinationPath)) {
                            $imagePath = $newImagePath;
                        }
                    }
                }

                $cw = Colourway::create([
                    'items_id' => $this->item->id,
                    'reference' => $ref,
                    'image' => $imagePath,
                    'sort_order' => $sort,
                ]);
                $created[] = $cw->reference;
            }

            // Refresh item to load new colourways and notify
            $this->item->refresh();
            $this->colourwayPdf = null;
            if (count($created) > 0) {
                session()->flash('message', 'Created ' . count($created) . ' colourway(s).');
            } else {
                session()->flash('message', 'No new colourways created.');
            }
        } finally {
            // Cleanup temp files
            foreach ([
                $pdfPath,
                $outPrefix . '.codes.txt',
                $outPrefix . '.codes.csv',
            ] as $p) {
                if ($p && is_file($p)) {
                    @unlink($p);
                }
            }
        }
    }

    private function extractPdfText(string $absolutePath): ?string
    {
        // Preferred: Smalot\PdfParser if installed
        if (class_exists('Smalot\\PdfParser\\Parser')) {
            try {
                $parser = new \Smalot\PdfParser\Parser();
                $pdf = $parser->parseFile($absolutePath);
                return $pdf->getText();
            } catch (\Throwable $e) {
                // fall through
            }
        }

        // Fallback: pdftotext CLI (Poppler) with -layout to preserve visual order
        $pdftotext = trim((string) @shell_exec('command -v pdftotext'));
        if ($pdftotext !== '') {
            $cmd = escapeshellcmd($pdftotext) . ' -layout -nopgbrk ' . escapeshellarg($absolutePath) . ' -';
            $output = @shell_exec($cmd);
            if (is_string($output) && $output !== '') {
                return $output;
            }
        }

        return null;
    }

    /**
     * Use Poppler pdftotext -bbox to capture word-level positions and
     * return colour codes in strict top-to-bottom, left-to-right order.
     */
    private function extractColourCodesWithBbox(string $absolutePath): array
    {
        $pdftotext = trim((string) @shell_exec('command -v pdftotext'));
        if ($pdftotext === '') {
            return [];
        }

        $cmd = escapeshellcmd($pdftotext) . ' -bbox ' . escapeshellarg($absolutePath) . ' -';
        $xml = @shell_exec($cmd);
        if (!is_string($xml) || trim($xml) === '') {
            return [];
        }

        // Parse XML; structure: <doc><page ...><word xMin="..." yMin="...">TEXT</word>...</page>...
        libxml_use_internal_errors(true);
        $sx = simplexml_load_string($xml);
        if ($sx === false) {
            return [];
        }

        $words = [];
        foreach ($sx->page as $page) {
            foreach ($page->word as $word) {
                $text = trim((string) $word);
                if ($text === '') {
                    continue;
                }
                $attrs = $word->attributes();
                $x = isset($attrs['xMin']) ? (float) $attrs['xMin'] : 0.0;
                $y = isset($attrs['yMin']) ? (float) $attrs['yMin'] : 0.0;
                $words[] = [ 't' => $text, 'x' => $x, 'y' => $y ];
            }
        }

        if (empty($words)) {
            return [];
        }

        // Sort top-to-bottom (y asc), then left-to-right (x asc)
        usort($words, function ($a, $b) {
            if (abs($a['y'] - $b['y']) > 0.5) { // small tolerance
                return $a['y'] <=> $b['y'];
            }
            return $a['x'] <=> $b['x'];
        });

        $results = [];
        foreach ($words as $w) {
            $t = strtoupper($w['t']);
            // Accept canonical colour code patterns only
            if (preg_match('/^#?[0-9A-F]{6}$/', $t)) { // hex colours
                $results[] = $t;
                continue;
            }
            // Optional grid labels like A1, B3 if you want to track order landmarks
            if (preg_match('/^[A-Z][0-9]{1,2}$/', $t)) {
                $results[] = $t;
                continue;
            }
            // Optional alnum code like NAV-001, only if looks like a reference (two+ letters and digits with hyphen)
            if (preg_match('/^[A-Z]{2,}[0-9-]+$/', $t)) {
                $results[] = $t;
            }
        }

        return $results;
    }

    /**
     * Heuristic colour code extraction preserving line/column order.
     * - Prefer hex colours like #A1B2C3 or A1B2C3
     * - Otherwise collect uppercase tokens with digits (SKU-like)
     */
    private function extractColourCodesFromText(string $text): array
    {
        $lines = preg_split('/\r\n|\r|\n/', $text) ?: [];
        $results = [];

        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') {
                continue;
            }

            // 1) Hex colours with #
            if (preg_match_all('/#([0-9A-Fa-f]{3,8})\b/', $line, $m)) {
                foreach ($m[0] as $hit) {
                    $results[] = strtoupper($hit);
                }
                continue;
            }

            // 2) Hex colours without # but 6 hex chars only (reduce noise)
            if (preg_match_all('/\b[0-9A-Fa-f]{6}\b/', $line, $m2)) {
                foreach ($m2[0] as $hit) {
                    $results[] = strtoupper($hit);
                }
            }

            // 3) Grid labels like A1, B3 etc - keep as markers if needed
            if (preg_match_all('/\b[A-Z]{1}[0-9]{1,2}\b/', $line, $m3)) {
                foreach ($m3[0] as $hit) {
                    $results[] = $hit; // may help verify ordering later
                }
            }

            // 4) Fallback: tokens like ABC-123, ABC123, but stricter to reduce noise
            if (preg_match_all('/\b[A-Z]{2,}[0-9][A-Z0-9-]*\b/', $line, $m4)) {
                foreach ($m4[0] as $hit) {
                    $results[] = strtoupper($hit);
                }
            }
        }

        return $results;
    }
    
    public function removeColourway($colourwayId)
    {
        $colourway = Colourway::findOrFail($colourwayId);
        $colourway->delete();
        
        // Refresh the item to get updated colourways
        $this->item->refresh();
        
        session()->flash('message', 'Colourway removed successfully!');
    }

    public function updateColourway($colourwayId, $field, $value)
    {
        $colourway = Colourway::findOrFail($colourwayId);
        $colourway->update([$field => $value]);
        
        session()->flash('message', 'Colourway updated successfully!');
    }

    public function updatedColourwayImages($value, $key)
    {
        // $key is the colourway ID, $value is the uploaded file
        if (!$value) return;

        $this->validate([
            "colourwayImages.{$key}" => 'image|max:2048',
        ]);

        $colourway = Colourway::findOrFail($key);
        
        // Delete old image if exists
        if ($colourway->image) {
            \Illuminate\Support\Facades\Storage::disk('public')->delete($colourway->image);
        }

        // Store new image
        $imagePath = $this->colourwayImages[$key]->store('colourways', 'public');
        
        // Update colourway with new image
        $colourway->update(['image' => $imagePath]);

        // Refresh the item to show updated colourway
        $this->item->refresh();
        
        // Clear the uploaded file from the array
        unset($this->colourwayImages[$key]);
        
        session()->flash('message', 'Colourway image updated successfully!');
    }

    public function uploadColourwayImage($colourwayId, $file = null)
    {
        // If no file parameter, try to get from tempColourwayImage
        if (!$file && $this->tempColourwayImage) {
            $file = $this->tempColourwayImage;
        }

        // Validate the uploaded file
        if (!$file) {
            session()->flash('error', 'No file provided.');
            return;
        }

        // For Livewire file uploads, we need to handle it differently
        if (is_string($file)) {
            // This is a temporary file path from Livewire
            $tempPath = storage_path('app/livewire-tmp/' . $file);
            if (!file_exists($tempPath)) {
                session()->flash('error', 'File not found.');
                return;
            }
            
            // Move the file to permanent storage
            $imagePath = 'colourways/' . $this->item->id . '_' . $colourwayId . '_' . time() . '.png';
            $destinationPath = storage_path('app/public/' . $imagePath);
            
            // Ensure directory exists
            $dir = dirname($destinationPath);
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
            
            if (!copy($tempPath, $destinationPath)) {
                session()->flash('error', 'Failed to save image.');
                return;
            }
            
            // Clean up temp file
            unlink($tempPath);
        } else {
            // Handle direct file upload (from drag/drop/paste)
            if (!$file->isValid()) {
                session()->flash('error', 'Invalid file uploaded.');
                return;
            }

            // Check file type and size
            if (!$file->getMimeType() || !str_starts_with($file->getMimeType(), 'image/')) {
                session()->flash('error', 'Please upload a valid image file.');
                return;
            }

            if ($file->getSize() > 2048 * 1024) { // 2MB in bytes
                session()->flash('error', 'Image file is too large. Maximum size is 2MB.');
                return;
            }

            // Store new image
            $imagePath = $file->store('colourways', 'public');
        }

        $colourway = Colourway::findOrFail($colourwayId);
        
        // Delete old image if exists
        if ($colourway->image) {
            \Illuminate\Support\Facades\Storage::disk('public')->delete($colourway->image);
        }

        // Update colourway with new image
        $colourway->update(['image' => $imagePath]);

        // Refresh the item to show updated colourway
        $this->item->refresh();
        
        // Clear temp file
        $this->tempColourwayImage = null;
        
        session()->flash('message', 'Image uploaded successfully!');
    }
    
    public function render()
    {
        $seasons = Season::where('is_active', true)
                        ->orderBy('sort_order')
                        ->orderBy('name')
                        ->get();
                        
        $categories = Category::where('is_active', true)
                             ->orderBy('sort_order')
                             ->orderBy('name')
                             ->get();
        
        return view('livewire.admin.item-detail', [
            'seasons' => $seasons,
            'categories' => $categories,
        ])
            ->layout('layouts.admin')
            ->title('Item Details');
    }
}
