<?php

namespace App\Livewire\Admin;

use App\Models\Item;
use App\Models\Season;
use App\Models\Category;
use App\Models\Colourway;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;

class ItemDetail extends Component
{
    use WithPagination, WithFileUploads;
    
    public Item $item;
    public $editing = false;
    
    // Form fields for editing
    public $name;
    public $slug;
    public $description;
    public $shortDescription;
    public $seasonsId;
    public $categoriesId;
    public $price;
    public $sku;
    public $isActive;
    public $images;
    public $noColumns;
    
    // Specifications management
    public $specifications = [];
    public $newSpecName = '';
    public $newSpecValue = '';
    
    // Colourways management
    public $newColourwayName = '';
    public $newColourwayReference = '';
    public $newColourwayColorCode = '';
    public $newColourwayImage;
    
    protected $rules = [
        'name' => 'required|string|max:255',
        'slug' => 'required|string|max:255|unique:items,slug',
        'description' => 'nullable|string',
        'shortDescription' => 'nullable|string',
        'seasonsId' => 'required|exists:seasons,id',
        'categoriesId' => 'required|exists:categories,id',
        'price' => 'nullable|numeric|min:0',
        'sku' => 'nullable|string|max:255|unique:items,sku',
        'isActive' => 'boolean',
        'noColumns' => 'required|integer|min:1|max:6',
    ];
    
    public function mount($itemId)
    {
        $this->item = Item::with(['season', 'category', 'colourways'])->findOrFail($itemId);
        $this->loadFormData();
    }
    
    public function loadFormData()
    {
        $this->name = $this->item->name;
        $this->slug = $this->item->slug;
        $this->description = $this->item->description;
        $this->shortDescription = $this->item->short_description;
        $this->seasonsId = $this->item->seasons_id;
        $this->categoriesId = $this->item->categories_id;
        $this->price = $this->item->price;
        $this->sku = $this->item->sku;
        $this->isActive = $this->item->is_active;
        $this->images = $this->item->images ?? [];
        $this->noColumns = $this->item->no_columns;
        $this->specifications = $this->item->specifications ?? [];
    }
    
    public function toggleEdit()
    {
        if ($this->editing) {
            $this->loadFormData(); // Reset form data
        }
        $this->editing = !$this->editing;
    }
    
    public function save()
    {
        // Update validation rules to exclude current item from unique checks
        $this->rules['slug'] = 'required|string|max:255|unique:items,slug,' . $this->item->id;
        $this->rules['sku'] = 'nullable|string|max:255|unique:items,sku,' . $this->item->id;
        
        $this->validate();
        
        $this->item->update([
            'name' => $this->name,
            'slug' => $this->slug,
            'description' => $this->description,
            'short_description' => $this->shortDescription,
            'seasons_id' => $this->seasonsId,
            'categories_id' => $this->categoriesId,
            'price' => $this->price,
            'sku' => $this->sku,
            'is_active' => $this->isActive,
            'images' => $this->images,
            'specifications' => $this->specifications,
            'no_columns' => $this->noColumns,
        ]);
        
        $this->editing = false;
        session()->flash('message', 'Item updated successfully!');
    }
    
    public function cancel()
    {
        $this->loadFormData();
        $this->editing = false;
    }
    
    // Specifications management
    public function addSpecification()
    {
        if ($this->newSpecName && $this->newSpecValue) {
            $this->specifications[] = [
                'name' => $this->newSpecName,
                'value' => $this->newSpecValue,
            ];
            $this->newSpecName = '';
            $this->newSpecValue = '';
        }
    }
    
    public function removeSpecification($index)
    {
        unset($this->specifications[$index]);
        $this->specifications = array_values($this->specifications);
    }
    
    // Colourways management
    public function addColourway()
    {
        $this->validate([
            'newColourwayName' => 'required|string|max:255',
            'newColourwayReference' => 'nullable|string|max:255',
            'newColourwayColorCode' => 'nullable|string|max:7',
            'newColourwayImage' => 'nullable|image|max:2048',
        ]);

        $imagePath = null;
        if ($this->newColourwayImage) {
            $imagePath = $this->newColourwayImage->store('colourways', 'public');
        }

        Colourway::create([
            'items_id' => $this->item->id,
            'name' => $this->newColourwayName,
            'reference' => $this->newColourwayReference,
            'color_code' => $this->newColourwayColorCode,
            'image' => $imagePath,
            'sort_order' => $this->item->colourways()->count(),
        ]);

        $this->newColourwayName = '';
        $this->newColourwayReference = '';
        $this->newColourwayColorCode = '';
        $this->newColourwayImage = null;

        // Refresh the item to get updated colourways
        $this->item->refresh();
        
        session()->flash('message', 'Colourway added successfully!');
    }
    
    public function removeColourway($colourwayId)
    {
        $colourway = Colourway::findOrFail($colourwayId);
        $colourway->delete();
        
        // Refresh the item to get updated colourways
        $this->item->refresh();
        
        session()->flash('message', 'Colourway removed successfully!');
    }

    public function updateColourway($colourwayId, $field, $value)
    {
        $colourway = Colourway::findOrFail($colourwayId);
        $colourway->update([$field => $value]);
        
        session()->flash('message', 'Colourway updated successfully!');
    }
    
    public function render()
    {
        $seasons = Season::where('is_active', true)
                        ->orderBy('sort_order')
                        ->orderBy('name')
                        ->get();
                        
        $categories = Category::where('is_active', true)
                             ->orderBy('sort_order')
                             ->orderBy('name')
                             ->get();
        
        return view('livewire.admin.item-detail', [
            'seasons' => $seasons,
            'categories' => $categories,
        ])
            ->layout('layouts.admin')
            ->title('Item Details');
    }
}
