<?php

namespace App\Services;

use App\Models\Receipt;
use App\Services\Matching\ClassificationService;
use App\Services\Matching\ReceiptMatchingService;
use App\Services\OCR\ReceiptOCRService;
use App\Services\Analysis\DiscountAnalysisService;
use App\Services\AI\ReceiptDeduplicationService;
use App\Services\Matching\GroupedReceiptMatchingService;

class ReceiptProcessingService
{
    public function __construct(
        private ReceiptOCRService $ocr,
        private ClassificationService $classification,
        private ReceiptMatchingService $matching,
        private DiscountAnalysisService $discountAnalysis,
        private ReceiptDeduplicationService $deduplication,
        private GroupedReceiptMatchingService $groupedMatching,
    ) {}

    public function process(Receipt $receipt): void
    {
        // Step 1: Extract data from the receipt image using OCR
        $this->ocr->extractReceiptData($receipt);
        
        // Refresh the receipt to get the updated data and lines
        $receipt->refresh();
        
        // Step 2: Check for duplicates and group related receipts BEFORE matching
        $group = $this->deduplication->processReceipt($receipt);
        
        // Step 3: Classify the receipt and its lines (VAT analysis will be done later during finance submission)
        $this->classification->classifyReceipt($receipt);
        
        // Process receipt lines for classification only (no VAT analysis yet)
        $this->processLinesForClassification($receipt);
        
        // Analyze discounts and link them to line items
        $this->discountAnalysis->analyzeDiscounts($receipt);
        
        // Step 4: Match to statement transactions
        if ($group) {
            // If receipt was grouped, match the entire group
            $this->groupedMatching->attemptMatchForGroup($group);
        } else {
            // If not grouped, match individual receipt
            $this->matching->attemptMatchForReceipt($receipt);
        }
    }

    /**
     * Process receipt lines for classification only (no VAT analysis)
     * This is much faster as it batches all lines into a single API call
     */
    private function processLinesForClassification(Receipt $receipt): void
    {
        if ($receipt->lines->isEmpty()) {
            return;
        }

        // Batch classify all lines in a single API call for much better performance
        $this->classification->classifyReceiptLinesBatch($receipt->lines);
    }
}


