<?php

namespace App\Services;

use App\Models\Receipt;
use App\Services\Matching\ClassificationService;
use App\Services\Matching\ReceiptMatchingService;
use App\Services\OCR\ReceiptOCRService;
use App\Services\Analysis\DiscountAnalysisService;
use App\Services\AI\ReceiptDeduplicationService;
use App\Services\Matching\GroupedReceiptMatchingService;

class ReceiptProcessingService
{
    public function __construct(
        private ReceiptOCRService $ocr,
        private ClassificationService $classification,
        private ReceiptMatchingService $matching,
        private DiscountAnalysisService $discountAnalysis,
        private ReceiptDeduplicationService $deduplication,
        private GroupedReceiptMatchingService $groupedMatching,
    ) {}

    public function process(Receipt $receipt): void
    {
        // This method is now deprecated - use the new 3-stage workflow instead
        // This method only does OCR and classification for individual receipts
        // Deduplication and matching should be handled by the new job system
        
        // Step 1: Extract data from the receipt image using OCR
        $this->ocr->extractReceiptData($receipt);
        
        // Refresh the receipt to get the updated data and lines
        $receipt->refresh();
        
        // Step 2: Classify the receipt and its lines (VAT analysis will be done later during finance submission)
        $this->classification->classifyReceipt($receipt);
        
        // Process receipt lines for classification only (no VAT analysis yet)
        $this->processLinesForClassification($receipt);
        
        // Analyze discounts and link them to line items
        $this->discountAnalysis->analyzeDiscounts($receipt);
        
        // NOTE: Deduplication and matching are now handled by separate jobs
        // in the new 3-stage workflow: OCR -> Deduplication -> Matching
    }

    /**
     * Process receipt lines for classification only (no VAT analysis)
     * This is much faster as it batches all lines into a single API call
     */
    private function processLinesForClassification(Receipt $receipt): void
    {
        if ($receipt->lines->isEmpty()) {
            return;
        }

        // Batch classify all lines in a single API call for much better performance
        $this->classification->classifyReceiptLinesBatch($receipt->lines);
    }
}


