<?php

namespace App\Services;

use App\Models\Receipt;
use App\Services\Matching\ClassificationService;
use App\Services\VAT\UkVatService;
use App\Services\Matching\ReceiptMatchingService;
use App\Services\OCR\ReceiptOCRService;
use App\Services\Analysis\DiscountAnalysisService;
use App\Services\CombinedLineAnalysisService;
use App\Jobs\ProcessReceiptLine;
use Illuminate\Support\Facades\Bus;

class ReceiptProcessingService
{
    public function __construct(
        private ReceiptOCRService $ocr,
        private ClassificationService $classification,
        private UkVatService $vat,
        private ReceiptMatchingService $matching,
        private DiscountAnalysisService $discountAnalysis,
    ) {}

    public function process(Receipt $receipt): void
    {
        // First extract data from the receipt image using OCR
        $this->ocr->extractReceiptData($receipt);
        
        // Refresh the receipt to get the updated data and lines
        $receipt->refresh();
        
        // Then classify the receipt and its lines (VAT analysis will be done later during finance submission)
        $this->classification->classifyReceipt($receipt);
        
        // Process receipt lines for classification only (no VAT analysis yet)
        $this->processLinesForClassification($receipt);
        
        // Analyze discounts and link them to line items
        $this->discountAnalysis->analyzeDiscounts($receipt);
        
        // Finally attempt to match the receipt to statement transactions
        $this->matching->attemptMatchForReceipt($receipt);
    }

    /**
     * Process receipt lines for classification only (no VAT analysis)
     * This is much faster as it only does one API call per line instead of two
     */
    private function processLinesForClassification(Receipt $receipt): void
    {
        if ($receipt->lines->isEmpty()) {
            return;
        }

        // Process each line for classification only
        foreach ($receipt->lines as $line) {
            $this->classification->classifyReceiptLine($line);
        }
    }
}


