<?php

namespace App\Services;

use App\Models\Receipt;
use App\Services\Matching\ClassificationService;
use App\Services\VAT\UkVatService;
use App\Services\Matching\ReceiptMatchingService;
use App\Services\OCR\ReceiptOCRService;
use App\Services\Analysis\DiscountAnalysisService;
use App\Jobs\ProcessReceiptLine;
use Illuminate\Support\Facades\Bus;

class ReceiptProcessingService
{
    public function __construct(
        private ReceiptOCRService $ocr,
        private ClassificationService $classification,
        private UkVatService $vat,
        private ReceiptMatchingService $matching,
        private DiscountAnalysisService $discountAnalysis,
    ) {}

    public function process(Receipt $receipt): void
    {
        // First extract data from the receipt image using OCR
        $this->ocr->extractReceiptData($receipt);
        
        // Refresh the receipt to get the updated data and lines
        $receipt->refresh();
        
        // Then classify the receipt and its lines
        $this->classification->classifyReceipt($receipt);
        
        // Process receipt lines in parallel for much faster processing
        $this->processLinesInParallel($receipt);
        
        // Analyze discounts and link them to line items
        $this->discountAnalysis->analyzeDiscounts($receipt);
        
        // Finally attempt to match the receipt to statement transactions
        $this->matching->attemptMatchForReceipt($receipt);
    }

    /**
     * Process receipt lines in parallel batches for much faster processing
     */
    private function processLinesInParallel(Receipt $receipt): void
    {
        if ($receipt->lines->isEmpty()) {
            return;
        }

        // Create jobs for each line
        $lineJobs = $receipt->lines->map(fn($line) => new ProcessReceiptLine($line->id))->toArray();
        
        // Process lines in batches of 3 to avoid overwhelming the API
        $batches = array_chunk($lineJobs, 3);
        
        foreach ($batches as $batch) {
            Bus::batch($batch)
                ->name("Process Receipt Lines for Receipt {$receipt->id}")
                ->allowFailures()
                ->onQueue('lines')
                ->dispatch();
        }
    }
}


