<?php

namespace App\Livewire\Admin;

use App\Models\Category;
use Livewire\Component;
use Livewire\WithPagination;

class CategoryDetail extends Component
{
    use WithPagination;
    
    public Category $category;
    public $editing = false;
    
    // Form fields for editing
    public $name;
    public $slug;
    public $description;
    public $isActive;
    public $sortOrder;
    
    protected $rules = [
        'name' => 'required|string|max:255',
        'slug' => 'required|string|max:255|unique:categories,slug',
        'description' => 'nullable|string',
        'isActive' => 'boolean',
        'sortOrder' => 'required|integer|min:0',
    ];
    
    public function mount($categoryId)
    {
        $this->category = Category::findOrFail($categoryId);
        $this->loadFormData();
        
        // Auto-enable editing for newly created categories
        if ($this->category->name === 'New Category') {
            $this->editing = true;
        }
    }
    
    public function loadFormData()
    {
        $this->name = $this->category->name;
        $this->slug = $this->category->slug;
        $this->description = $this->category->description;
        $this->isActive = $this->category->is_active;
        $this->sortOrder = $this->category->sort_order;
    }
    
    public function toggleEdit()
    {
        if ($this->editing) {
            $this->loadFormData(); // Reset form data
        }
        $this->editing = !$this->editing;
    }
    
    public function save()
    {
        // Update validation rules to exclude current category from unique check
        $this->rules['slug'] = 'required|string|max:255|unique:categories,slug,' . $this->category->id;
        
        $this->validate();
        
        $this->category->update([
            'name' => $this->name,
            'slug' => $this->slug,
            'description' => $this->description,
            'is_active' => $this->isActive,
            'sort_order' => $this->sortOrder,
        ]);
        
        $this->editing = false;
        session()->flash('message', 'Category updated successfully!');
    }
    
    public function cancel()
    {
        $this->loadFormData();
        $this->editing = false;
    }
    
    public function render()
    {
        // Get items for this category (when Item model is created)
        $items = collect(); // Empty collection for now
        
        return view('livewire.admin.category-detail', [
            'items' => $items
        ])
            ->layout('layouts.admin')
            ->title('Category Details');
    }
}
