# Home Page Customization System

## Overview

The Home Page Customization System allows photographers to personalize their gallery home pages by choosing from predefined layouts and filling in customizable fields. This provides a professional, branded landing page for each photographer's gallery.

## Key Concepts

### 1. Layout Templates
Layout templates are predefined page designs created and managed by superusers. Each layout has:
- **Layout Key**: Unique identifier (e.g., `classic`, `modern`)
- **Name**: Display name shown to users
- **Template File**: EJS file that renders the layout (stored in `views/home-layouts/`)
- **Description**: Helps users choose the right layout
- **Status**: Active/inactive
- **Default Flag**: One layout can be marked as default for new users

### 2. Customization Elements
Elements are individual fields that users can customize. Examples:
- **hero_title**: Main heading text
- **hero_image**: Background image URL
- **about_text**: About section content
- **instagram_url**: Social media link
- **accent_color**: Theme color

Each element has:
- **Element Key**: Unique identifier
- **Element Type**: text, textarea, url, color, image, checkbox, etc.
- **Label**: Display name for users
- **Description**: Help text
- **Default Value**: Fallback if not customized
- **Status**: Active/inactive

### 3. Layout-Element Mapping
Layouts use different combinations of elements. For example:
- **Classic Layout**: Uses hero, about, contact, and social elements
- **Modern Layout**: Uses hero, about, and social elements (no contact section)
- **Minimal Layout**: Uses only hero and social elements

## Database Schema

### Tables Created

1. **customization_elements** - Available customization fields
2. **layout_templates** - Layout designs
3. **layout_elements** - Maps elements to layouts
4. **user_home_customizations** - User's customized values
5. **users.home_layout_id** - User's selected layout

## Setup Instructions

### Step 1: Run Migration

```bash
# Set your database URL
export DATABASE_URL='postgresql://username:password@localhost:5432/vybephoto'

# Run the migration
chmod +x run-migration-008.sh
./run-migration-008.sh
```

The migration automatically creates:
- 4 default layouts (classic, modern, minimal, full_width)
- 20 default customization elements
- Pre-configured element mappings for each layout

### Step 2: Verify Installation

1. Log in as a superuser at `app.vybephoto.com/superuser`
2. Navigate to **Layouts** - you should see 4 layouts
3. Navigate to **Elements** - you should see 20 elements
4. Click "Edit" on any layout to see which elements it uses

## Superuser Management

### Managing Layouts

**Access:** `/superuser/customization/layouts`

**Creating a Layout:**
1. Click "+ Create New Layout"
2. Fill in:
   - Layout Key (unique, lowercase, e.g., `elegant`)
   - Name (display name, e.g., "Elegant Portfolio")
   - Description (helps users choose)
   - Template File (EJS filename, e.g., `home-elegant.ejs`)
   - Display Order (sorting)
   - Active status
   - Default flag (only one can be default)
3. Submit the form
4. Click "Edit" on the new layout to add elements

**Adding Elements to a Layout:**
1. Go to `/superuser/customization/layouts`
2. Click "Edit" on a layout
3. Scroll to "Add Elements" section
4. For each element:
   - Select section name (e.g., "hero", "about")
   - Set display order
   - Mark as required if needed
   - Click "Add to Layout"

**Removing Elements:**
- In the layout edit view, click "Remove" next to any element

### Managing Elements

**Access:** `/superuser/customization/elements`

**Creating an Element:**
1. Click "+ Create New Element"
2. Fill in:
   - Element Key (unique, lowercase, underscores, e.g., `footer_text`)
   - Label (user-facing name, e.g., "Footer Text")
   - Element Type (text, textarea, url, color, image, checkbox)
   - Description (help text for users)
   - Default Value (optional)
   - Display Order
   - Active status
3. Submit the form

**Element Types:**
- **text**: Short text input (headlines, names)
- **textarea**: Long text input (descriptions, paragraphs)
- **url**: URL input with validation (social links, CTAs)
- **color**: Color picker (theme colors)
- **image**: Image URL input (backgrounds, photos)
- **checkbox**: True/false toggle (feature flags)
- **rich_text**: Rich text editor (future enhancement)

### Best Practices

1. **Naming Conventions:**
   - Element keys: `section_field` (e.g., `hero_title`, `about_image`)
   - Section names: Consistent across layouts (hero, about, contact, social, cta)

2. **Default Values:**
   - Provide sensible defaults so pages look good even without customization
   - Use placeholders that demonstrate the expected content

3. **Required Fields:**
   - Mark essential elements as required (e.g., hero_title)
   - Don't over-require - let users customize progressively

4. **Element Reusability:**
   - Create elements that work across multiple layouts
   - Example: `instagram_url` can be used in any layout with social links

## User Experience

### For Photographers

1. **Access Customization:**
   - Navigate to `/admin/home` or click "🎨 Edit Home" in admin nav
   
2. **Choose Layout:**
   - View available layouts with descriptions
   - Click to select (auto-saves and reloads)

3. **Customize Content:**
   - Fill in fields grouped by section
   - Required fields marked with red asterisk
   - See help text under each field
   - Use default values as guidance

4. **Save & Preview:**
   - Click "💾 Save Changes"
   - Click "🌐 View Site" to see live result

### What Users See

- Grouped fields by section (Hero, About, Contact, etc.)
- Clear labels and help text
- Input types appropriate to content (color picker, textarea, etc.)
- Default values pre-filled
- Validation for required fields

## Creating Custom Layouts

### File Structure

Create new layout template in `views/home-layouts/`:

```
views/
  home-layouts/
    home-classic.ejs      (default layouts)
    home-modern.ejs
    home-minimal.ejs
    home-fullwidth.ejs
    home-elegant.ejs      (your custom layout)
```

### Layout Template Format

```ejs
<!DOCTYPE html>
<html>
<head>
  <title><%= siteConfig.siteName %></title>
  <!-- Include fonts, styles -->
  <style>
    /* Custom styles for this layout */
    /* Access primary color: <%= siteConfig.primaryColor %> */
  </style>
</head>
<body>
  <!-- Use customization values -->
  <h1><%= customizations.hero_title || 'Default Title' %></h1>
  <p><%= customizations.hero_subtitle %></p>
  
  <% if (customizations.about_text) { %>
    <section>
      <%= customizations.about_text %>
    </section>
  <% } %>
  
  <!-- Recent albums (if enabled) -->
  <% if (customizations.show_recent_albums === 'true' && recentAlbums.length > 0) { %>
    <% recentAlbums.forEach(album => { %>
      <a href="/albums/<%= album.id %>">
        <img src="<%= album.cover_photo_url %>" alt="<%= album.name %>">
      </a>
    <% }); %>
  <% } %>
  
  <%- include('../_footer') %>
</body>
</html>
```

### Available Variables

**In Layout Templates:**
- `customizations` - Object with element_key => value
- `siteConfig` - Tenant branding (siteName, primaryColor, logoUrl, etc.)
- `recentAlbums` - Array of public albums (if show_recent_albums enabled)
- `req` - Request object with tenant info

### Adding Your Layout

1. Create template file: `views/home-layouts/home-elegant.ejs`
2. Go to superuser panel: `/superuser/customization/layouts`
3. Click "+ Create New Layout"
4. Fill in:
   - Layout Key: `elegant`
   - Name: `Elegant Portfolio`
   - Template File: `home-elegant.ejs`
5. Submit, then click "Edit" to add elements

## Troubleshooting

### Layout Not Showing
- Check layout is marked as "Active"
- Verify template file exists in `views/home-layouts/`
- Check server logs for rendering errors

### Elements Not Saving
- Verify migration ran successfully
- Check element IDs in form match database IDs
- Look for JavaScript errors in browser console

### Default Layout Not Working
- Ensure at least one layout is marked as default
- Check layout is active
- Verify template file is valid

### User's Homepage Shows Error
- System falls back to original `index.ejs` on errors
- Check server logs for specific error
- Verify all referenced customization elements exist

## API Reference

### Superuser Functions

Located in `src/superuser.js`:

**Elements:**
- `getAllCustomizationElements(activeOnly)` - List elements
- `getCustomizationElementById(elementId)` - Get single element
- `createCustomizationElement(adminUserId, elementData, req)` - Create element
- `updateCustomizationElement(adminUserId, elementId, elementData, req)` - Update element
- `deleteCustomizationElement(adminUserId, elementId, req)` - Delete element

**Layouts:**
- `getAllLayoutTemplates(activeOnly)` - List layouts
- `getLayoutTemplateById(layoutId)` - Get single layout
- `getLayoutTemplateWithElements(layoutId)` - Get layout with its elements
- `createLayoutTemplate(adminUserId, layoutData, req)` - Create layout
- `updateLayoutTemplate(adminUserId, layoutId, layoutData, req)` - Update layout
- `deleteLayoutTemplate(adminUserId, layoutId, req)` - Delete layout

**Layout-Element Mapping:**
- `addElementToLayout(adminUserId, layoutId, elementId, options, req)` - Add element
- `removeElementFromLayout(adminUserId, layoutId, elementId, req)` - Remove element
- `updateLayoutElement(adminUserId, layoutId, elementId, options, req)` - Update mapping

### Routes

**Superuser:**
- `GET /superuser/customization/elements` - List elements
- `GET /superuser/customization/elements/new` - Create element form
- `GET /superuser/customization/elements/:id/edit` - Edit element form
- `POST /superuser/customization/elements/create` - Create element
- `POST /superuser/customization/elements/:id/update` - Update element
- `POST /superuser/customization/elements/:id/delete` - Delete element
- `GET /superuser/customization/layouts` - List layouts
- `GET /superuser/customization/layouts/new` - Create layout form
- `GET /superuser/customization/layouts/:id/edit` - Edit layout (with elements)
- `POST /superuser/customization/layouts/create` - Create layout
- `POST /superuser/customization/layouts/:id/update` - Update layout
- `POST /superuser/customization/layouts/:id/delete` - Delete layout
- `POST /superuser/customization/layouts/:layoutId/elements/:elementId/add` - Add element to layout
- `POST /superuser/customization/layouts/:layoutId/elements/:elementId/remove` - Remove element from layout

**User Admin:**
- `GET /admin/home` - Home customization interface
- `POST /admin/home/save` - Save customizations

## Future Enhancements

Potential improvements:

1. **Visual Layout Picker:** Show preview images of layouts
2. **Live Preview:** Real-time preview as users edit
3. **Image Upload:** Direct image uploads instead of URLs
4. **Rich Text Editor:** WYSIWYG editing for text areas
5. **Section Reordering:** Drag-and-drop section ordering
6. **Template Builder:** Visual template builder for superusers
7. **Layout Cloning:** Duplicate existing layouts as starting point
8. **Version History:** Track changes to customizations
9. **A/B Testing:** Test different layouts/content
10. **Import/Export:** Share layouts between instances

## Support

For issues or questions:
1. Check this documentation
2. Review server logs for errors
3. Check database for data integrity
4. Verify all migrations have run
5. Ensure template files exist and are valid EJS

## Migration History

- **Migration 008** (2025-01-11): Initial home customization system
  - Created 4 tables
  - Added 20 default elements
  - Added 4 default layouts
  - Configured element-layout mappings

