<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Tests\TestCase;

class RouteTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test user
        User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);
    }

    /** @test */
    public function home_route_is_accessible()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
    }

    /** @test */
    public function login_route_is_accessible()
    {
        $response = $this->get('/login');
        
        $response->assertStatus(200);
    }

    /** @test */
    public function register_route_is_accessible()
    {
        $response = $this->get('/register');
        
        $response->assertStatus(200);
    }

    /** @test */
    public function home_route_redirects_guests_to_login()
    {
        $response = $this->get('/home');
        
        $response->assertRedirect('/login');
    }

    /** @test */
    public function home_route_is_accessible_to_authenticated_users()
    {
        $user = User::where('email', 'test@example.com')->first();
        
        $response = $this->actingAs($user)->get('/home');
        
        $response->assertStatus(200);
    }

    /** @test */
    public function login_post_route_works()
    {
        $response = $this->post('/login', [
            'email' => 'test@example.com',
            'password' => 'password',
        ]);

        $response->assertRedirect('/home');
    }

    /** @test */
    public function register_post_route_works()
    {
        $response = $this->post('/register', [
            'name' => 'New User',
            'email' => 'newuser@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ]);

        $response->assertRedirect('/home');
    }

    /** @test */
    public function logout_route_works()
    {
        $user = User::where('email', 'test@example.com')->first();
        
        $response = $this->actingAs($user)->post('/logout');
        
        $response->assertRedirect('/');
    }

    /** @test */
    public function password_reset_route_is_accessible()
    {
        $response = $this->get('/password/reset');
        
        $response->assertStatus(200);
    }

    /** @test */
    public function password_request_route_is_accessible()
    {
        $response = $this->get('/password/request');
        
        $response->assertStatus(200);
    }

    /** @test */
    public function routes_have_correct_names()
    {
        $this->assertEquals(url('/'), route('home'));
    }

    /** @test */
    public function authentication_routes_are_registered()
    {
        $routes = \Route::getRoutes();
        
        $routeNames = collect($routes)->map(function ($route) {
            return $route->getName();
        })->toArray();

        $this->assertContains('login', $routeNames);
        $this->assertContains('register', $routeNames);
        $this->assertContains('logout', $routeNames);
    }

    /** @test */
    public function guest_middleware_works_on_login_route()
    {
        $user = User::where('email', 'test@example.com')->first();
        
        $response = $this->actingAs($user)->get('/login');
        
        $response->assertRedirect('/home');
    }

    /** @test */
    public function guest_middleware_works_on_register_route()
    {
        $user = User::where('email', 'test@example.com')->first();
        
        $response = $this->actingAs($user)->get('/register');
        
        $response->assertRedirect('/home');
    }

    /** @test */
    public function auth_middleware_works_on_home_route()
    {
        $response = $this->get('/home');
        
        $response->assertRedirect('/login');
    }

    /** @test */
    public function logout_route_requires_authentication()
    {
        $response = $this->post('/logout');
        
        $response->assertRedirect('/login');
    }

    /** @test */
    public function routes_use_correct_http_methods()
    {
        // Test GET routes
        $this->get('/')->assertStatus(200);
        $this->get('/login')->assertStatus(200);
        $this->get('/register')->assertStatus(200);
        
        // Test POST routes
        $this->post('/login', [
            'email' => 'test@example.com',
            'password' => 'password',
        ])->assertRedirect('/home');
        
        $this->post('/register', [
            'name' => 'New User',
            'email' => 'newuser@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ])->assertRedirect('/home');
    }

    /** @test */
    public function routes_return_correct_views()
    {
        $this->get('/')->assertViewIs('welcome');
        $this->get('/login')->assertViewIs('auth.login');
        $this->get('/register')->assertViewIs('auth.register');
    }

    /** @test */
    public function home_route_returns_correct_view_for_authenticated_users()
    {
        $user = User::where('email', 'test@example.com')->first();
        
        $response = $this->actingAs($user)->get('/home');
        
        $response->assertViewIs('home');
    }

    /** @test */
    public function routes_handle_csrf_protection()
    {
        $response = $this->post('/login', [
            'email' => 'test@example.com',
            'password' => 'password',
            // Missing CSRF token
        ]);

        $response->assertStatus(419); // CSRF token mismatch
    }

    /** @test */
    public function routes_handle_invalid_requests()
    {
        $response = $this->post('/login', [
            'email' => 'invalid-email',
            'password' => 'short',
        ]);

        $response->assertSessionHasErrors(['email', 'password']);
    }

    /** @test */
    public function routes_handle_missing_parameters()
    {
        $response = $this->post('/login', [
            // Missing required parameters
        ]);

        $response->assertSessionHasErrors(['email', 'password']);
    }

    /** @test */
    public function routes_handle_duplicate_registration()
    {
        // First registration
        $this->post('/register', [
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ]);

        // Second registration with same email
        $response = $this->post('/register', [
            'name' => 'Another User',
            'email' => 'test@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ]);

        $response->assertSessionHasErrors(['email']);
    }
}