const express = require('express');
const stripeService = require('./stripe-service');
const stripeWebhook = require('./stripe-webhook');

const router = express.Router();

/**
 * Middleware to check if user is authenticated (multi-tenant mode)
 * For single-tenant mode, this needs to be adapted
 */
function requireAuth(req, res, next) {
  if (!req.user || !req.user.id) {
    return res.status(401).json({ error: 'Authentication required' });
  }
  next();
}

/**
 * GET /admin/subscription
 * Display subscription management page
 */
router.get('/admin/subscription', requireAuth, async (req, res) => {
  try {
    const db = req.db; // Assuming db is attached to req
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).send('User not found');
    }
    
    // Get usage statistics
    const usage = await db.getUserUsage(user.id);
    
    // Get album count
    const albums = await db.listAlbumsByUser(user.id);
    const albumCount = albums.length;
    
    res.render('admin/subscription', {
      pageTitle: 'Subscription',
      user,
      usage,
      albumCount,
      req,
    });
  } catch (error) {
    console.error('[SUBSCRIPTION] Error loading subscription page:', error);
    res.status(500).send('Error loading subscription');
  }
});

/**
 * POST /admin/subscription/change
 * Change subscription tier
 */
router.post('/admin/subscription/change', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    const newTier = req.body.tier;
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    // Validate tier
    const validTiers = ['free', 'starter', 'professional', 'business'];
    if (!validTiers.includes(newTier)) {
      return res.status(400).json({ error: 'Invalid tier' });
    }
    
    // Handle downgrade to free
    if (newTier === 'free') {
      if (user.stripeSubscriptionId) {
        // Cancel Stripe subscription
        await stripeService.cancelSubscription(user.stripeSubscriptionId);
      }
      
      // Update user to free tier
      await db.updateUser(user.id, {
        subscriptionTier: 'free',
        subscriptionStatus: 'active',
        stripeSubscriptionId: null,
      });
      
      return res.redirect('/admin/subscription?success=downgraded');
    }
    
    // Handle upgrade/downgrade with Stripe
    if (!stripeService.isConfigured()) {
      return res.status(500).json({ error: 'Stripe not configured' });
    }
    
    // If user doesn't have a Stripe customer ID, create one
    if (!user.stripeCustomerId) {
      const customerId = await stripeService.createCustomer(
        user.email,
        user.username,
        user.fullName
      );
      
      await db.updateUser(user.id, {
        stripeCustomerId: customerId,
      });
      
      user.stripeCustomerId = customerId;
    }
    
    // If user has an existing subscription, update it
    if (user.stripeSubscriptionId) {
      await stripeService.updateSubscription(user.stripeSubscriptionId, newTier);
      
      // Update user tier (webhook will update other fields)
      await db.updateUser(user.id, {
        subscriptionTier: newTier,
      });
      
      return res.redirect('/admin/subscription?success=changed');
    }
    
    // Create new subscription with checkout
    const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
    const baseUrl = `${protocol}://${req.get('host')}`;
    const successUrl = `${baseUrl}/admin/subscription?success=subscribed`;
    const cancelUrl = `${baseUrl}/admin/subscription?canceled=true`;
    
    const checkoutUrl = await stripeService.createCheckoutSession(
      user.stripeCustomerId,
      newTier,
      successUrl,
      cancelUrl
    );
    
    return res.redirect(checkoutUrl);
  } catch (error) {
    console.error('[SUBSCRIPTION] Error changing subscription:', error);
    res.redirect('/admin/subscription?error=change_failed');
  }
});

/**
 * POST /admin/subscription/cancel
 * Cancel subscription at period end
 */
router.post('/admin/subscription/cancel', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    if (!user.stripeSubscriptionId) {
      return res.redirect('/admin/subscription?error=no_subscription');
    }
    
    // Cancel subscription at period end
    const result = await stripeService.cancelSubscription(user.stripeSubscriptionId);
    
    // Update user status
    await db.updateUser(user.id, {
      subscriptionStatus: 'canceled',
      subscriptionEndsAt: result.cancelAt,
    });
    
    res.redirect('/admin/subscription?success=canceled');
  } catch (error) {
    console.error('[SUBSCRIPTION] Error canceling subscription:', error);
    res.redirect('/admin/subscription?error=cancel_failed');
  }
});

/**
 * POST /admin/subscription/reactivate
 * Reactivate a cancelled subscription
 */
router.post('/admin/subscription/reactivate', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    if (!user.stripeSubscriptionId) {
      return res.redirect('/admin/subscription?error=no_subscription');
    }
    
    // Reactivate subscription
    await stripeService.reactivateSubscription(user.stripeSubscriptionId);
    
    // Update user status
    await db.updateUser(user.id, {
      subscriptionStatus: 'active',
    });
    
    res.redirect('/admin/subscription?success=reactivated');
  } catch (error) {
    console.error('[SUBSCRIPTION] Error reactivating subscription:', error);
    res.redirect('/admin/subscription?error=reactivate_failed');
  }
});

/**
 * POST /admin/subscription/payment-portal
 * Redirect to Stripe billing portal
 */
router.post('/admin/subscription/payment-portal', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    if (!user.stripeCustomerId) {
      return res.redirect('/admin/subscription?error=no_customer');
    }
    
    // Create billing portal session
    const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
    const returnUrl = `${protocol}://${req.get('host')}/admin/subscription`;
    
    const portalUrl = await stripeService.createBillingPortalSession(
      user.stripeCustomerId,
      returnUrl
    );
    
    res.redirect(portalUrl);
  } catch (error) {
    console.error('[SUBSCRIPTION] Error creating portal session:', error);
    res.redirect('/admin/subscription?error=portal_failed');
  }
});

/**
 * POST /webhooks/stripe
 * Handle Stripe webhook events
 * This endpoint should NOT use requireAuth middleware
 */
router.post('/webhooks/stripe', express.raw({ type: 'application/json' }), async (req, res) => {
  const signature = req.headers['stripe-signature'];
  const webhookSecret = process.env.STRIPE_WEBHOOK_SECRET;
  
  if (!webhookSecret) {
    console.error('[STRIPE WEBHOOK] Webhook secret not configured');
    return res.status(500).json({ error: 'Webhook secret not configured' });
  }
  
  try {
    // Verify webhook signature
    const event = stripeService.constructWebhookEvent(
      req.body,
      signature,
      webhookSecret
    );
    
    // Handle the event
    const db = req.db; // Assuming db is attached to req
    await stripeWebhook.handleWebhook(db, event);
    
    res.json({ received: true });
  } catch (error) {
    console.error('[STRIPE WEBHOOK] Error processing webhook:', error);
    res.status(400).json({ error: `Webhook Error: ${error.message}` });
  }
});

/**
 * GET /api/subscription/status
 * Get current subscription status (for AJAX requests)
 */
router.get('/api/subscription/status', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    const usage = await db.getUserUsage(user.id);
    
    res.json({
      tier: user.subscriptionTier,
      status: user.subscriptionStatus,
      trialEndsAt: user.trialEndsAt,
      subscriptionEndsAt: user.subscriptionEndsAt,
      usage,
    });
  } catch (error) {
    console.error('[SUBSCRIPTION] Error getting status:', error);
    res.status(500).json({ error: 'Failed to get subscription status' });
  }
});

module.exports = router;

