const express = require('express');
const stripeService = require('./stripe-service');
const stripeWebhook = require('./stripe-webhook');

const router = express.Router();

/**
 * Middleware to check if user is authenticated (multi-tenant mode)
 * For single-tenant mode, this needs to be adapted
 */
function requireAuth(req, res, next) {
  if (!req.user || !req.user.id) {
    return res.status(401).json({ error: 'Authentication required' });
  }
  next();
}

/**
 * GET /admin/subscription
 * Display subscription management page
 */
router.get('/admin/subscription', requireAuth, async (req, res) => {
  try {
    const db = req.db; // Assuming db is attached to req
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).send('User not found');
    }
    
    // Get usage statistics
    const usage = await db.getUserUsage(user.id);
    
    // Get album count
    const albums = await db.listAlbumsByUser(user.id);
    const albumCount = albums.length;
    
    res.render('admin/subscription', {
      pageTitle: 'Subscription',
      user,
      usage,
      albumCount,
      req,
    });
  } catch (error) {
    console.error('[SUBSCRIPTION] Error loading subscription page:', error);
    res.status(500).send('Error loading subscription');
  }
});

/**
 * POST /admin/subscription/change
 * Change subscription tier with Stripe Checkout for upgrades/downgrades
 */
router.post('/admin/subscription/change', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    const newTier = req.body.tier;
    
    console.log(`[SUBSCRIPTION] User ${user.id} requesting change from ${user.subscriptionTier} to ${newTier}`);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    // Don't allow changing to the same tier
    if (user.subscriptionTier === newTier) {
      return res.redirect('/admin/subscription?error=same_tier');
    }
    
    // For paid tiers, validate that the tier is active and available
    if (newTier !== 'free') {
      const tierConfig = await stripeService.getTierConfig(newTier);
      
      if (!tierConfig) {
        console.error(`[SUBSCRIPTION] Invalid tier requested: ${newTier}`);
        return res.redirect('/admin/subscription?error=invalid_tier');
      }
      
      // Check if tier is active (available for new subscriptions)
      if (!tierConfig.isActive) {
        console.error(`[SUBSCRIPTION] Tier ${newTier} is not available for new subscriptions`);
        return res.redirect('/admin/subscription?error=tier_unavailable');
      }
    }
    
    // Handle downgrade to free
    if (newTier === 'free') {
      if (user.stripeSubscriptionId) {
        // Cancel Stripe subscription at period end
        await stripeService.cancelSubscription(user.stripeSubscriptionId);
        console.log(`[SUBSCRIPTION] Cancelled subscription ${user.stripeSubscriptionId} for user ${user.id}`);
      }
      
      // Update user to free tier
      await db.updateUser(user.id, {
        subscriptionTier: 'free',
        subscriptionStatus: 'canceled',
        stripeSubscriptionId: null,
      });
      
      return res.redirect('/admin/subscription?success=downgraded');
    }
    
    // Handle upgrade/downgrade with Stripe
    if (!(await stripeService.isConfigured())) {
      console.error('[SUBSCRIPTION] Stripe not configured');
      return res.status(500).json({ error: 'Stripe not configured' });
    }
    
    // If user doesn't have a Stripe customer ID, create one
    if (!user.stripeCustomerId) {
      console.log(`[SUBSCRIPTION] Creating Stripe customer for user ${user.id}`);
      const customerId = await stripeService.createCustomer(
        user.email,
        user.username,
        user.fullName
      );
      
      await db.updateUser(user.id, {
        stripeCustomerId: customerId,
      });
      
      user.stripeCustomerId = customerId;
    }
    
    // Build URLs for checkout
    const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
    const baseUrl = `${protocol}://${req.get('host')}`;
    const successUrl = `${baseUrl}/admin/subscription/upgrade-complete?session_id={CHECKOUT_SESSION_ID}`;
    const cancelUrl = `${baseUrl}/admin/subscription?canceled=true`;
    
    // If user has an existing paid subscription, cancel it and create new checkout
    // This ensures proper proration and refund through Stripe Checkout
    if (user.stripeSubscriptionId) {
      console.log(`[SUBSCRIPTION] User has existing subscription ${user.stripeSubscriptionId}, will cancel after new subscription is active`);
      
      // Create checkout session for new subscription
      // We'll cancel the old subscription in the success handler after payment
      const checkoutUrl = await stripeService.createCheckoutSession(
        user.stripeCustomerId,
        newTier,
        successUrl,
        cancelUrl,
        false, // No trial for upgrades
        user.stripeSubscriptionId // Pass old subscription ID for cancellation
      );
      
      console.log(`[SUBSCRIPTION] Created upgrade checkout session for user ${user.id}`);
      return res.redirect(checkoutUrl);
    }
    
    // Create new subscription with checkout (first time subscribing)
    console.log(`[SUBSCRIPTION] Creating first subscription for user ${user.id}`);
    const checkoutUrl = await stripeService.createCheckoutSession(
      user.stripeCustomerId,
      newTier,
      successUrl,
      cancelUrl,
      false // No trial for plan changes
    );
    
    return res.redirect(checkoutUrl);
  } catch (error) {
    console.error('[SUBSCRIPTION] Error changing subscription:', error);
    res.redirect('/admin/subscription?error=change_failed');
  }
});

/**
 * GET /admin/subscription/upgrade-complete
 * Handle successful upgrade checkout completion
 */
router.get('/admin/subscription/upgrade-complete', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    const sessionId = req.query.session_id;
    
    if (!sessionId) {
      return res.redirect('/admin/subscription?error=no_session');
    }
    
    console.log(`[SUBSCRIPTION] Processing upgrade completion for user ${user.id}, session ${sessionId}`);
    
    // Retrieve the checkout session
    const session = await stripeService.stripe.checkout.sessions.retrieve(sessionId, {
      expand: ['subscription']
    });
    
    if (!session || !session.subscription) {
      console.error('[SUBSCRIPTION] Invalid session or no subscription found');
      return res.redirect('/admin/subscription?error=invalid_session');
    }
    
    const newSubscriptionId = session.subscription.id;
    const newTier = session.metadata.tier;
    const oldSubscriptionId = session.metadata.old_subscription_id;
    
    console.log(`[SUBSCRIPTION] New subscription: ${newSubscriptionId}, Old subscription: ${oldSubscriptionId}, New tier: ${newTier}`);
    
    // Update user with new subscription
    await db.updateUser(user.id, {
      subscriptionTier: newTier,
      subscriptionStatus: session.subscription.status,
      stripeSubscriptionId: newSubscriptionId,
    });
    
    // If there was an old subscription, cancel it immediately and create a refund
    if (oldSubscriptionId && oldSubscriptionId !== '') {
      try {
        console.log(`[SUBSCRIPTION] Cancelling old subscription ${oldSubscriptionId} immediately with proration`);
        
        // Cancel the old subscription immediately (not at period end)
        // This will create a proration credit for the unused time
        await stripeService.stripe.subscriptions.cancel(oldSubscriptionId, {
          prorate: true, // Create credit for unused time
          invoice_now: true, // Create invoice immediately with the credit
        });
        
        console.log(`[SUBSCRIPTION] Old subscription ${oldSubscriptionId} cancelled with prorated refund`);
      } catch (cancelError) {
        console.error(`[SUBSCRIPTION] Error cancelling old subscription:`, cancelError);
        // Continue anyway - the new subscription is active
      }
    }
    
    console.log(`[SUBSCRIPTION] Upgrade complete for user ${user.id}`);
    res.redirect('/admin/subscription?success=upgraded');
  } catch (error) {
    console.error('[SUBSCRIPTION] Error processing upgrade completion:', error);
    res.redirect('/admin/subscription?error=upgrade_failed');
  }
});

/**
 * POST /admin/subscription/cancel
 * Cancel subscription at period end
 */
router.post('/admin/subscription/cancel', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    if (!user.stripeSubscriptionId) {
      return res.redirect('/admin/subscription?error=no_subscription');
    }
    
    // Cancel subscription at period end
    const result = await stripeService.cancelSubscription(user.stripeSubscriptionId);
    
    // Update user status
    await db.updateUser(user.id, {
      subscriptionStatus: 'canceled',
      subscriptionEndsAt: result.cancelAt,
    });
    
    res.redirect('/admin/subscription?success=canceled');
  } catch (error) {
    console.error('[SUBSCRIPTION] Error canceling subscription:', error);
    res.redirect('/admin/subscription?error=cancel_failed');
  }
});

/**
 * POST /admin/subscription/reactivate
 * Reactivate a cancelled subscription
 */
router.post('/admin/subscription/reactivate', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    if (!user.stripeSubscriptionId) {
      return res.redirect('/admin/subscription?error=no_subscription');
    }
    
    // Reactivate subscription
    await stripeService.reactivateSubscription(user.stripeSubscriptionId);
    
    // Update user status
    await db.updateUser(user.id, {
      subscriptionStatus: 'active',
    });
    
    res.redirect('/admin/subscription?success=reactivated');
  } catch (error) {
    console.error('[SUBSCRIPTION] Error reactivating subscription:', error);
    res.redirect('/admin/subscription?error=reactivate_failed');
  }
});

/**
 * POST /admin/subscription/payment-portal
 * Redirect to Stripe billing portal
 */
router.post('/admin/subscription/payment-portal', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    if (!user.stripeCustomerId) {
      return res.redirect('/admin/subscription?error=no_customer');
    }
    
    // Create billing portal session
    const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
    const returnUrl = `${protocol}://${req.get('host')}/admin/subscription`;
    
    const portalUrl = await stripeService.createBillingPortalSession(
      user.stripeCustomerId,
      returnUrl
    );
    
    res.redirect(portalUrl);
  } catch (error) {
    console.error('[SUBSCRIPTION] Error creating portal session:', error);
    res.redirect('/admin/subscription?error=portal_failed');
  }
});

/**
 * POST /webhooks/stripe
 * Handle Stripe webhook events
 * This endpoint should NOT use requireAuth middleware
 */
router.post('/webhooks/stripe', express.raw({ type: 'application/json' }), async (req, res) => {
  const signature = req.headers['stripe-signature'];
  const webhookSecret = process.env.STRIPE_WEBHOOK_SECRET;
  
  if (!webhookSecret) {
    console.error('[STRIPE WEBHOOK] Webhook secret not configured');
    return res.status(500).json({ error: 'Webhook secret not configured' });
  }
  
  try {
    // Verify webhook signature
    const event = stripeService.constructWebhookEvent(
      req.body,
      signature,
      webhookSecret
    );
    
    // Handle the event
    const db = req.db; // Assuming db is attached to req
    await stripeWebhook.handleWebhook(db, event);
    
    res.json({ received: true });
  } catch (error) {
    console.error('[STRIPE WEBHOOK] Error processing webhook:', error);
    res.status(400).json({ error: `Webhook Error: ${error.message}` });
  }
});

/**
 * GET /api/subscription/status
 * Get current subscription status (for AJAX requests)
 */
router.get('/api/subscription/status', requireAuth, async (req, res) => {
  try {
    const db = req.db;
    const user = await db.getUserById(req.user.id);
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    const usage = await db.getUserUsage(user.id);
    
    res.json({
      tier: user.subscriptionTier,
      status: user.subscriptionStatus,
      trialEndsAt: user.trialEndsAt,
      subscriptionEndsAt: user.subscriptionEndsAt,
      usage,
    });
  } catch (error) {
    console.error('[SUBSCRIPTION] Error getting status:', error);
    res.status(500).json({ error: 'Failed to get subscription status' });
  }
});

module.exports = router;

